/*
  **********************************************************************
  *
  *     Copyright 1999, 2000 Creative Labs, Inc.
  *
  **********************************************************************
  *
  *     Date                 Author               Summary of changes
  *     ----                 ------               ------------------
  *     October 20, 1999     Andrew de Quincey    Rewrote and extended
  *                          Lucien Murray-Pitts  original incomplete 
  *                                               driver.
  *
  *     April 18, 1999       Andrew Veliath       Original Driver
  *                                               implementation
  *
  **********************************************************************
  *
  *     This program is free software; you can redistribute it and/or
  *     modify it under the terms of the GNU General Public License as
  *     published by the Free Software Foundation; either version 2 of
  *     the License, or (at your option) any later version.
  *
  *     This program is distributed in the hope that it will be useful,
  *     but WITHOUT ANY WARRANTY; without even the implied warranty of
  *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  *     GNU General Public License for more details.
  *
  *     You should have received a copy of the GNU General Public
  *     License along with this program; if not, write to the Free
  *     Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139,
  *     USA.
  *
  **********************************************************************
  */

/**
 *
 * Driver for the Brooktree/Rockwell/Conexant BT865 TV encoder chip
 * High level convenience functions
 *
 */

#include <linux/delay.h>
#include <linux/errno.h>
#include <bt865.h>


/**
 *
 * Initialises the bt865 chip
 *
 * @param instance bt865 instance to use
 *
 */

extern int bt865_init(bt865_t* instance)
{
  int tmp;

  // reset the chip
  bt865_set_reg(instance, 0xA6, 0x80);
  
  // set TXHS[7:0] to 0
  bt865_set_reg(instance, 0xAC, 0);

  // set TXHE[7:0] to 0
  bt865_set_reg(instance, 0xAE, 0);

  // set TXHS[10:8], TXHE[10:8], LUMADLY[1:0] to 0
  bt865_set_reg(instance, 0xB0, 0);

  // basically, disable teletext
  bt865_set_reg(instance, 0xB2, 0);

  // sets a reserved bit in register 0xBC!!!
  bt865_set_reg(instance, 0xBC, 0x10);

  // noninterlaced mode, setup off, among other things
  bt865_set_reg(instance, 0xCC, 0x42);

  // normal video mode, ESTATUS = 0
  bt865_set_reg(instance, 0xCE, 2);
  
  // enable WSS/CGMS in second field
  bt865_set_reg(instance, 0xA0, 0x80);

  // OK
  return(0);
}
  
  

/**
 *
 * Put the bt865 into PAL mode
 *
 * @param instance bt865 instance to use
 *
 * @return 0 on success, <0 on failure
 *
 */

extern int bt865_set_PAL_mode(bt865_t* instance) 
{
  // PAL mode
  bt865_set_reg(instance, 0xCC, 0xE4);

  // but not PAL-N
  bt865_set_reg(instance, 0xD0, 0);
  
  // OK
  return(0);
}


/**
 *
 * Put the bt865 into NTSC mode
 *
 * @param instance bt865 instance to use
 *
 * @return 0 on success, <0 on failure
 *
 */

extern int bt865_set_NTSC_mode(bt865_t* instance) 
{
  // NTSC mode
  bt865_set_reg(instance, 0xCC, 0x80);

  // turn PAL-N bit off for good measure!
  bt865_set_reg(instance, 0xD0, 0);
  
  // OK
  return(0);
}



/**
 *
 * Set the macrovision mode.
 *
 * @param instance bt865 instance to use
 * @param macrovisionMode (one of BT865_MACROVISION_OFF, BT865_MACROVISION_1, BT865_MACROVISION_2,
 *                         BT865_MACROVISION_3, BT865_MACROVISION_AGC)
 * @param tvMode Current TV mode (one of BT865_PAL_MODE, or BT865_NTSC_MODE)
 *
 * @return 0 on success, <0 on failure
 *
 */

extern int bt865_set_macrovision_mode(bt865_t* instance, int macrovisionMode, int tvMode)
{
  int tmp;


  // set bt865 into normal video
  bt865_set_reg(instance, 0xCE,2);

  // get BT865 mode. if it's a bt864, return now, since it doesn't support macrovision
  tmp = bt865_get_reg(instance, BT865_READBACK);
  if (tmp != 0xa0) {
    
    return(0);
  }
  
  // if we're in PAL mode, there are really only two macrovision modes (off, and mode 4)
  if ((tvMode == BT865_PAL_MODE) && (macrovisionMode != BT865_MACROVISION_OFF)) {
    
    macrovisionMode = BT865_MACROVISION_AGC;
  }

  // OK, do it
  switch(macrovisionMode) {
  case BT865_MACROVISION_OFF:
    
    bt865_set_reg(instance, 0xd8, 0);
    break;
    
  case BT865_MACROVISION_1:
    
    bt865_set_reg(instance, 0xd8, 0);
    bt865_set_reg(instance, 0xdA, 0x60);
    bt865_set_reg(instance, 0xdC, 0xF);
    bt865_set_reg(instance, 0xdE, 0xF);
    bt865_set_reg(instance, 0xE0, 0x0);
    bt865_set_reg(instance, 0xE2, 0);
    bt865_set_reg(instance, 0xE4, 0xFC);
    bt865_set_reg(instance, 0xE6, 3);
    bt865_set_reg(instance, 0xE8, 0xB9);
    bt865_set_reg(instance, 0xEA, 0x6D);
    bt865_set_reg(instance, 0xEC, 0xb6);
    bt865_set_reg(instance, 0xEE, 0xd5);
    bt865_set_reg(instance, 0xF0, 0xb0);
    bt865_set_reg(instance, 0xF2, 0x72);
    bt865_set_reg(instance, 0xF4, 0x0D);
    bt865_set_reg(instance, 0xF6, 0xff);
    bt865_set_reg(instance, 0xF8, 0x2c);
    bt865_set_reg(instance, 0xFA, 0xd0);
    bt865_set_reg(instance, 0xD8, 0x36);
    break;
    
  case BT865_MACROVISION_2:
    
    bt865_set_reg(instance, 0xd8, 0);
    bt865_set_reg(instance, 0xdA, 0x60);
    bt865_set_reg(instance, 0xdC, 0xF);
    bt865_set_reg(instance, 0xdE, 0xF);
    bt865_set_reg(instance, 0xE0, 0);
    bt865_set_reg(instance, 0xE2, 0);
    bt865_set_reg(instance, 0xE4, 0xFC);
    bt865_set_reg(instance, 0xE6, 3);
    bt865_set_reg(instance, 0xE8, 0xB9);
    bt865_set_reg(instance, 0xEA, 0x6D);
    bt865_set_reg(instance, 0xEC, 0x3c);
    bt865_set_reg(instance, 0xEE, 0xd1);
    bt865_set_reg(instance, 0xF0, 0x32);
    bt865_set_reg(instance, 0xF2, 0xd2);
    bt865_set_reg(instance, 0xF4, 0x0D);
    bt865_set_reg(instance, 0xF6, 0xff);
    bt865_set_reg(instance, 0xF8, 0x2c);
    bt865_set_reg(instance, 0xFA, 0xd0);
    bt865_set_reg(instance, 0xD8, 0x3e);
    break;

  case BT865_MACROVISION_3:
    
    bt865_set_reg(instance, 0xd8, 0);
    bt865_set_reg(instance, 0xdA, 0x60);
    bt865_set_reg(instance, 0xdC, 0xF);
    bt865_set_reg(instance, 0xdE, 0xF);
    bt865_set_reg(instance, 0xE0, 0);
    bt865_set_reg(instance, 0xE2, 0);
    bt865_set_reg(instance, 0xE4, 0xFC);
    bt865_set_reg(instance, 0xE6, 3);
    bt865_set_reg(instance, 0xE8, 0xB9);
    bt865_set_reg(instance, 0xEA, 0x6D);
    bt865_set_reg(instance, 0xEC, 0xB6);
    bt865_set_reg(instance, 0xEE, 0xd5);
    bt865_set_reg(instance, 0xF0, 0xb0);
    bt865_set_reg(instance, 0xF2, 0x72);
    bt865_set_reg(instance, 0xF4, 0x0D);
    bt865_set_reg(instance, 0xF6, 0xff);
    bt865_set_reg(instance, 0xF8, 0x2c);
    bt865_set_reg(instance, 0xFA, 0xd0);
    bt865_set_reg(instance, 0xD8, 0x3e);
    break;

  case BT865_MACROVISION_AGC:
    
    bt865_set_reg(instance, 0xd8, 0);
    bt865_set_reg(instance, 0xdA, 0x60);
    bt865_set_reg(instance, 0xdC, 0x7E);
    bt865_set_reg(instance, 0xdE, 0xFE);
    bt865_set_reg(instance, 0xE0, 0x54);
    bt865_set_reg(instance, 0xE2, 1);
    bt865_set_reg(instance, 0xE4, 0xFF);
    bt865_set_reg(instance, 0xE6, 1);
    bt865_set_reg(instance, 0xE8, 0xD5);
    bt865_set_reg(instance, 0xEA, 0x73);
    bt865_set_reg(instance, 0xEC, 0xa8);
    bt865_set_reg(instance, 0xEE, 0x62);
    bt865_set_reg(instance, 0xF0, 0x55);
    bt865_set_reg(instance, 0xF2, 0xa4);
    bt865_set_reg(instance, 0xF4, 0x5);
    bt865_set_reg(instance, 0xF6, 0x55);
    bt865_set_reg(instance, 0xF8, 0x27);
    bt865_set_reg(instance, 0xFA, 0x40);
    bt865_set_reg(instance, 0xD8, 0x36);
    break;
    
  default:
    
    return(-EINVAL);
  }
  
  // OK
  return(0);
}
