/*
  **********************************************************************
  *
  *     Copyright 1999, 2000 Creative Labs, Inc.
  *
  **********************************************************************
  *
  *     Date                 Author               Summary of changes
  *     ----                 ------               ------------------
  *     October 20, 1999     Andrew de Quincey    Rewrote and extended
  *                          Lucien Murray-Pitts  original incomplete 
  *                                               driver.
  *
  *     April 18, 1999       Andrew Veliath       Original Driver
  *                                               implementation
  *
  **********************************************************************
  *
  *     This program is free software; you can redistribute it and/or
  *     modify it under the terms of the GNU General Public License as
  *     published by the Free Software Foundation; either version 2 of
  *     the License, or (at your option) any later version.
  *
  *     This program is distributed in the hope that it will be useful,
  *     but WITHOUT ANY WARRANTY; without even the implied warranty of
  *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  *     GNU General Public License for more details.
  *
  *     You should have received a copy of the GNU General Public
  *     License along with this program; if not, write to the Free
  *     Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139,
  *     USA.
  *
  **********************************************************************
  */

/**
 *
 * Driver for the Creative DXR2 Mpeg decoder card
 *
 */


#ifndef __DXR2_H__
#define __DXR2_H__

#include <linux/types.h>
#include <linux/ioctl.h>
#include <linux/tqueue.h>
#include <anp82.h>
#include <bt865.h>
#include <pcm1723.h>
#include <tc6807af.h>
#include <vxp524.h>
#include <zivaDS.h>



// *******************************************************************
// useful defines

// full name of the supported thing
#define DXR2_FULLNAME			"Creative DXR2 Mpeg decoder"

// log name of the driver
#define DXR2_LOGNAME			"DXR2"

// these control entry to code modifying deferred_queue stuff
#define DXR2_ENTER_CRITICAL(DXR2INST) while(!atomic_dec_and_test(&((DXR2INST)->semaphore))) { atomic_inc(&((DXR2INST)->semaphore)); schedule(); }
#define DXR2_EXIT_CRITICAL(DXR2INST) atomic_inc(&((DXR2INST)->semaphore))

// number of pages in each BM buffer (2^DXR2_PAGE_ORDER)
#define DXR2_PAGE_ORDER 3

#define DXR2_MAJOR 120

#define DXR2_MAX_DEFERRED_COMMANDS 5

// *******************************************************************
// IOCTL codes (this is temporary)

#define DXR2_IOC_MAGIC 'X'

#define DXR2_IOC_GET_REGION_CODE		_IO(DXR2_IOC_MAGIC, 0)
#define DXR2_IOC_SET_OUTPUT_TV_FORMAT		_IO(DXR2_IOC_MAGIC, 1)
#define DXR2_IOC_SET_SOURCE_VIDEO_FORMAT 	_IO(DXR2_IOC_MAGIC, 2)
#define DXR2_IOC_GET_CAPABILITIES		_IO(DXR2_IOC_MAGIC, 3)
#define DXR2_IOC_CLEAR_VIDEO			_IO(DXR2_IOC_MAGIC, 4)
#define DXR2_IOC_PAUSE				_IO(DXR2_IOC_MAGIC, 5)
#define DXR2_IOC_SET_AUDIO_VOLUME		_IO(DXR2_IOC_MAGIC, 6)
#define DXR2_IOC_SET_OUTPUT_ASPECT_RATIO	_IO(DXR2_IOC_MAGIC, 7)
#define DXR2_IOC_ABORT				_IO(DXR2_IOC_MAGIC, 8)
#define DXR2_IOC_STOP				_IO(DXR2_IOC_MAGIC, 9)
#define DXR2_IOC_ENABLE_SUBPICTURE		_IO(DXR2_IOC_MAGIC, 10)
#define DXR2_IOC_SLOW_FORWARDS			_IO(DXR2_IOC_MAGIC, 11)
#define DXR2_IOC_SLOW_BACKWARDS			_IO(DXR2_IOC_MAGIC, 12)
#define DXR2_IOC_SET_SOURCE_ASPECT_RATIO	_IO(DXR2_IOC_MAGIC, 13)
#define DXR2_IOC_SET_ASPECT_RATIO_MODE		_IO(DXR2_IOC_MAGIC, 14)
#define DXR2_IOC_SINGLE_STEP			_IO(DXR2_IOC_MAGIC, 15)
#define DXR2_IOC_REVERSE_PLAY			_IO(DXR2_IOC_MAGIC, 16)
#define DXR2_IOC_SET_SUBPICTURE_PALETTE		_IO(DXR2_IOC_MAGIC, 17)
#define DXR2_IOC_GET_CHALLENGE_KEY		_IO(DXR2_IOC_MAGIC, 18)
#define DXR2_IOC_SEND_CHALLENGE_KEY		_IO(DXR2_IOC_MAGIC, 19)
#define DXR2_IOC_GET_RESPONSE_KEY		_IO(DXR2_IOC_MAGIC, 20)
#define DXR2_IOC_SEND_RESPONSE_KEY		_IO(DXR2_IOC_MAGIC, 21)
#define DXR2_IOC_SEND_DISC_KEY			_IO(DXR2_IOC_MAGIC, 22)
#define DXR2_IOC_SEND_TITLE_KEY			_IO(DXR2_IOC_MAGIC, 23)
#define DXR2_IOC_SET_DECRYPTION_MODE		_IO(DXR2_IOC_MAGIC, 24)
#define DXR2_IOC_INIT_ZIVADS			_IO(DXR2_IOC_MAGIC, 25)
#define DXR2_IOC_SCAN_MODE			_IO(DXR2_IOC_MAGIC, 26)
#define DXR2_IOC_SET_MACROVISION_MODE		_IO(DXR2_IOC_MAGIC, 27)
#define DXR2_IOC_RESET 				_IO(DXR2_IOC_MAGIC, 28)
#define DXR2_IOC_SET_BITSTREAM_TYPE 		_IO(DXR2_IOC_MAGIC, 29)
#define DXR2_IOC_PLAY 				_IO(DXR2_IOC_MAGIC, 30)
#define DXR2_IOC_GET_STC      			_IO(DXR2_IOC_MAGIC, 31)
#define DXR2_IOC_SET_AUDIO_SAMPLE_FREQUENCY 	_IO(DXR2_IOC_MAGIC, 32)
#define DXR2_IOC_SET_AUDIO_DATA_WIDTH		_IO(DXR2_IOC_MAGIC, 33)
#define DXR2_IOC_IEC958_OUTPUT_MODE		_IO(DXR2_IOC_MAGIC, 34)
#define DXR2_IOC_SET_AC3_MODE			_IO(DXR2_IOC_MAGIC, 35)
#define DXR2_IOC_SELECT_AC3_VOICE		_IO(DXR2_IOC_MAGIC, 36)
#define DXR2_IOC_AUDIO_MUTE			_IO(DXR2_IOC_MAGIC, 37)
#define DXR2_IOC_SET_STEREO_MODE		_IO(DXR2_IOC_MAGIC, 38)
#define DXR2_IOC_SELECT_STREAM			_IO(DXR2_IOC_MAGIC, 39)
#define DXR2_IOC_HIGHLIGHT			_IO(DXR2_IOC_MAGIC, 40)
    


// *******************************************************************
// stuff for IOCTLS

// TV formats
#define DXR2_TVFORMAT_NTSC 0
#define DXR2_TVFORMAT_PAL  1

// video frequencies
#define DXR2_SRC_VIDEO_FREQ_30 0
#define DXR2_SRC_VIDEO_FREQ_25 1

// aspect ratios
#define DXR2_ASPECTRATIO_4_3 0
#define DXR2_ASPECTRATIO_16_9 1

// subpicture modes
#define DXR2_SUBPICTURE_OFF 0
#define DXR2_SUBPICTURE_ON 1

// rates for slow forwards & backwards
#define DXR2_SLOWRATE_2x 0
#define DXR2_SLOWRATE_3x 1
#define DXR2_SLOWRATE_4x 2
#define DXR2_SLOWRATE_5x 3
#define DXR2_SLOWRATE_6x 4

// CSS decryption modes supported
#define DXR2_CSSDECRMODE_OFF 0
#define DXR2_CSSDECRMODE_ON  1

// play modes
#define DXR2_PLAYMODE_STOPPED         0
#define DXR2_PLAYMODE_PAUSED          1
#define DXR2_PLAYMODE_SLOWFORWARDS    2
#define DXR2_PLAYMODE_SLOWBACKWARDS   3
#define DXR2_PLAYMODE_SINGLESTEP      4
#define DXR2_PLAYMODE_PLAY            5
#define DXR2_PLAYMODE_REVERSEPLAY     6
#define DXR2_PLAYMODE_SCAN            7

// for operation queue
#define DXR2_QUEUE_PAUSED             0
#define DXR2_QUEUE_SETVOLUME          1
#define DXR2_QUEUE_ENABLESUBPICTURE   2
#define DXR2_QUEUE_SCANMODE           3
#define DXR2_QUEUE_SELECTSTREAM       4
#define DXR2_QUEUE_SETMUTESTATUS      5
#define DXR2_QUEUE_HIGHLIGHT          6

// aspect ratio modes
#define DXR2_ASPECTRATIOMODE_NORMAL    0
#define DXR2_ASPECTRATIOMODE_PAN_SCAN  1
#define DXR2_ASPECTRATIOMODE_LETTERBOX 2

// macrovision modes
#define DXR2_MACROVISION_OFF 0
#define DXR2_MACROVISION_1   1
#define DXR2_MACROVISION_2   2
#define DXR2_MACROVISION_3   3
#define DXR2_MACROVISION_4   4

// bitstreams possibly present in files
#define DXR2_STREAM_VIDEO       0
#define DXR2_STREAM_SUBPICTURE  1
#define DXR2_STREAM_AUDIO_AC3   2
#define DXR2_STREAM_AUDIO_MPEG  3
#define DXR2_STREAM_AUDIO_LPCM  4
#define DXR2_STREAM_AUDIO_5 5

// bitstream types
#define DXR2_BITSTREAM_TYPE_MPEG_VOB       0
#define DXR2_BITSTREAM_TYPE_CDROM_VCD      1
#define DXR2_BITSTREAM_TYPE_MPEG_VCD       2
#define DXR2_BITSTREAM_TYPE_CDDA           3
#define DXR2_BITSTREAM_TYPE_4              4

// frequency of output audio data (to the pcm1723)
#define DXR2_AUDIO_FREQ_441    0
#define DXR2_AUDIO_FREQ_48     1
#define DXR2_AUDIO_FREQ_96     2
#define DXR2_AUDIO_FREQ_2205   3
#define DXR2_AUDIO_FREQ_32     4

// widths of output audio data (to the pcm1723)
#define DXR2_AUDIO_WIDTH_16    0
#define DXR2_AUDIO_WIDTH_20    1
#define DXR2_AUDIO_WIDTH_24    2

// play types
#define DXR2_PLAYTYPE_NORMAL 0
#define DXR2_PLAYTYPE_STILLSTOP 1

// iec-958 output types
#define DXR2_IEC958_DECODED    0
#define DXR2_IEC958_ENCODED    1

// AC3 modes
#define DXR2_AC3MODE_LR_STEREO          0
#define DXR2_AC3MODE_LR_STEREO_PROLOGIC 1
#define DXR2_AC3MODE_LR_MONOR           2

// AC3 voice configuration (for karaoke)
#define DXR2_AC3VOICE_NONE        0
#define DXR2_AC3VOICE_V1V2        1

// highlight actions
#define DXR2_BUTTONACTION_SELECT    0
#define DXR2_BUTTONACTION_UNHIGHLIGHT 1
#define DXR2_BUTTONACTION_ACTIVATE 2
#define DXR2_BUTTONACTION_ACTIVATE_SELECTED 3
#define DXR2_BUTTONACTION_4 4
#define DXR2_BUTTONACTION_5 5
#define DXR2_BUTTONACTION_6 6
#define DXR2_BUTTONACTION_7 7
#define DXR2_BUTTONACTION_8 8

// special buttons
#define DXR2_BUTTON_NONE    0
#define DXR2_BUTTON_UP     64
#define DXR2_BUTTON_DOWN   65
#define DXR2_BUTTON_LEFT   66
#define DXR2_BUTTON_RIGHT  67

// mute modes
#define DXR2_AUDIO_MUTE_ON     0
#define DXR2_AUDIO_MUTE_OFF    1

// stereo mode
#define DXR2_AUDIO_STEREO_NORMAL  0
#define DXR2_AUDIO_STEREO_MONOL   1
#define DXR2_AUDIO_STEREO_MONOR   2
#define DXR2_AUDIO_STEREO_REVERSE 3




// *******************************************************************
// Structures


typedef struct {

  // driver instances
  anp82_t* anp82Instance;
  bt865_t* bt865Instance;
  pcm1723_t* pcm1723Instance;
  tc6807af_t* tc6807afInstance;
  zivaDS_t* zivaDSInstance;
  vxp524_t* vxp524Instance;

  // base address values used for talking to some of the hardware
  unsigned long tc6807afBase;
  unsigned long zivaDSBase;
  unsigned long vxp524Base;
  unsigned long asicBase;

  // current value of the ASIC, since it's read only
  int asicValue;

  // BM buffer stuff
  int bmBuffer;
  int writeBuffer;
  unsigned long buffer[2];
  int bufferSize[2];
  int bufferCount[2];

  // is the HLI int enabled
  int hliFlag;

  // has the ziva been initialised
  int zivaDSInitialised;

  // semaphore so we don't start two BMs/process the deferred queue at the same time
  atomic_t semaphore;

  // wait queue
  struct wait_queue* waitQueue;
  
  // deferred commands
  int deferredCount;
  int deferredQueue[DXR2_MAX_DEFERRED_COMMANDS][4];

  // to hold data to transfer
  char* userBuffer;            // user supplied buffer
  int userBytesTransferred;      // number of bytes already transferred
  int userBufferSize;            // total amount of data in buffer

  // misc stuff we need to remember
  int currentOutputTvFormat;
  int currentZivaAudioDACMode;
  int currentSourceVideoFrequency;
  int currentSourceVideoXRes;
  int currentSourceVideoYRes;
  int currentPlayMode;
  int currentAudioVolume;
  int currentOutputAspectRatio;
  int currentSlowRate;
  int currentAspectRatioMode;
  int currentScanOp;
  int currentScanParam;
  int currentBitstreamType;
  int currentVideoStream;
  int currentSubPictureStream;
  int currentAudioStream;
  int currentAudioStreamType;
  int currentAudioMuteStatus;
  int currentAudioSampleFrequency;
  int currentAudioInputWidth;

} dxr2_t;

typedef struct {

  int arg;

} dxr2_oneArg_t;


typedef struct {

  int arg1;
  int arg2;

} dxr2_twoArg_t;

typedef struct {

  int arg1;
  int arg2;
  int arg3;

} dxr2_threeArg_t;


typedef struct {

  int uCodeLength;
  char uCode[0]; // allocate this structure to whatever length you need...

} dxr2_uCode_t;

typedef struct {

  int entries[16];

} dxr2_palette_t;

typedef struct {
  
  char key[10];

} dxr2_challengeKey_t;


typedef struct {
  
  char key[5];

} dxr2_responseKey_t;


typedef struct {
  
  char key[0x800];

} dxr2_discKey_t;


typedef struct {
  
  char key[6];

} dxr2_titleKey_t;



// *******************************************************************
// function declarations


// ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
// Driver maintenance functions


/**
 *
 * Create new DXR2 device
 *
 */

extern dxr2_t* dxr2_new();
		     


/**
 *
 * Destroy a dxr2 device (BURNY BURNY!!!!)
 *
 * @param instance DXR2 instance to use
 *
 */

extern void dxr2_free(dxr2_t* instance);


#endif
