/*	X Window Video for Linux Two Control Panel Tool
 *
 *	For testing a Video for Linux Two driver
 *
 *	This program was written by Bill Dirks.
 *	This program is in the public domain.
 *
 *	gcc -o vidpanel -L/usr/X11R6/lib/ -lXt -lXaw -Wall vidpanel.c
 */

/*  Set these according to your set-up */
#define MY_DEVICE "/dev/video0"


#include <sys/time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <sys/mman.h>
#include <errno.h>

/* These are needed to use the V4L2 driver */
#include <linux/fs.h>
#include <linux/kernel.h>
#include <linux/videodev2.h>  /* Video for Linux Two */

#include <X11/StringDefs.h>
#include <X11/Intrinsic.h>
#include <X11/Xaw/Command.h>
#include <X11/Xaw/Box.h>
#include <X11/Xaw/Scrollbar.h>
#include <X11/Xaw/MenuButton.h>
#include <X11/Xaw/SimpleMenu.h>
#include <X11/Xaw/SmeBSB.h>
#include <X11/Xaw/Toggle.h>

typedef struct
{
	int	index;
	int	ctrl_id;
	Widget	menuitem;
}	MenuEntry;
#define MAXMENU	12

typedef struct
{
	struct v4l2_queryctrl	qc;
	Widget			box;
	Widget			label;
	Widget			widget;
	Widget			menu;
	MenuEntry		menuentry[MAXMENU];
}	GuiCtrl;

typedef struct
{
	XtAppContext		xtapp;
	Widget			w_toplevel;
	Widget			w_mainbox;
	Widget			w_menubox;
	Widget			w_videobox;
	Widget			w_inputmb;
	Widget			w_inputmenu;
	Widget			w_quit;
	Widget			w_videodefault;
	Widget			w_device;
} PanelApp;


int			vid;
GuiCtrl			ctrl[12];
PanelApp		app;


#define SLIDERLEN	160
void init_all_controls();


void quit(Widget w, XtPointer client, XtPointer call)
{
	exit(0);
}

/*
 *  Scrollbar-type widgets for V4L2 INTEGER controls
 */

void
scroll_jump_proc(Widget w, XtPointer client, XtPointer call)
{
	struct v4l2_control	vc;
	GuiCtrl			*ctrl = (GuiCtrl *)client;
	float			percent = *(float *)call;

	vc.id = ctrl->qc.id;
	vc.value = ctrl->qc.minimum
		 + (int)(percent * (ctrl->qc.maximum - ctrl->qc.minimum + 1));
	if (vc.value > ctrl->qc.maximum)
		vc.value = ctrl->qc.maximum;
	ioctl(vid, VIDIOC_S_CTRL, &vc);
}

void
scroll_proc(Widget w, XtPointer client, XtPointer call)
{
	Arg		arg[1];
	XtArgVal	*p;
	int		d = (int)call;
	static float	f;

	XtVaGetValues(w, XtNtopOfThumb, &f, NULL);
	f += (float)d / (float)SLIDERLEN;
	if (f < 0) f = 0;
	if (f > 1) f = 1;
	
	p = (XtArgVal *)&f;
	if (sizeof(float) > sizeof(XtArgVal))
		XtSetArg(arg[0], XtNtopOfThumb, &f);
	else
		XtSetArg(arg[0], XtNtopOfThumb, *p);
	XtSetValues(w, arg, 1);
	scroll_jump_proc(w, client, &f);/* to change the control */
}

void
init_scroll_control(GuiCtrl *ctrl)
{
	struct v4l2_control	vc;
	int			d;
	static float		f;
	Arg			arg[1];
	XtArgVal		*p;

	/*  Initialize the thumb position  */
	vc.id = ctrl->qc.id;
	if (ioctl(vid, VIDIOC_G_CTRL, &vc))
		return;
	d = (SLIDERLEN * (vc.value - ctrl->qc.minimum)
	     + ((ctrl->qc.maximum - ctrl->qc.minimum) >> 1))
		/ (ctrl->qc.maximum - ctrl->qc.minimum);

	f = 0;
	p = (XtArgVal *)&f;
	if (sizeof(float) > sizeof(XtArgVal))
		XtSetArg(arg[0], XtNtopOfThumb, &f);
	else
		XtSetArg(arg[0], XtNtopOfThumb, *p);
	XtSetValues(ctrl->widget, arg, 1);
	scroll_proc(ctrl->widget, ctrl, (XtPointer)d);
}

void
add_scroll_control(GuiCtrl *ctrl)
{
	ctrl->widget =
		XtVaCreateManagedWidget("scroll",
					scrollbarWidgetClass,
					ctrl->box,
					XtNorientation, XtorientHorizontal,
					XtNlength, SLIDERLEN,
					XtNminimumThumb, 14,
					XtNthickness, 18,
					NULL);
	ctrl->label = 
		XtVaCreateManagedWidget("label",
					labelWidgetClass, ctrl->box,
					XtNlabel, ctrl->qc.name,
					XtNborderWidth, 0,
					NULL);
	XtAddCallback(ctrl->widget, XtNscrollProc, scroll_proc, ctrl);
	XtAddCallback(ctrl->widget, XtNjumpProc, scroll_jump_proc, ctrl);
}


/*
 *  Toggle-type widgets for V4L2 BOOLEAN controls
 */

void
boolproc(Widget w, XtPointer client, XtPointer call)
{
	GuiCtrl			*ctrl = (GuiCtrl *)client;
	int			d = (int)call;
	struct v4l2_control	vc;

	vc.id = ctrl->qc.id;
	vc.value = d;
	ioctl(vid, VIDIOC_S_CTRL, &vc);
}

void
init_bool_control(GuiCtrl *ctrl)
{
	struct v4l2_control	vc;
	Arg			arg[1];

	vc.id = ctrl->qc.id;
	if (ioctl(vid, VIDIOC_G_CTRL, &vc))
		return;
	XtSetArg(arg[0], XtNstate, vc.value);
	XtSetValues(ctrl->widget, arg, 1);
}

void
add_bool_control(GuiCtrl *ctrl)
{
	ctrl->widget =
		XtVaCreateManagedWidget("bool",
					toggleWidgetClass,
					ctrl->box,
					XtNlabel, ctrl->qc.name,
					XtNorientation, XtorientHorizontal,
					XtNwidth, SLIDERLEN,
					NULL);
	XtAddCallback(ctrl->widget, XtNcallback, boolproc, ctrl);
}


/*
 *  Menu-type widgets for V4L2 MENU controls
 */

#define VIDIN_ID 1010

void
menu_proc(Widget w, XtPointer client, XtPointer call)
{
	MenuEntry		*menuentry = (MenuEntry *)client;
	struct v4l2_control	vc;

	if (menuentry->ctrl_id == VIDIN_ID)
	{
		ioctl(vid, VIDIOC_S_INPUT, menuentry->index);
		init_all_controls();
		return;
	}
	vc.id = menuentry->ctrl_id;
	vc.value = menuentry->index;
	ioctl(vid, VIDIOC_S_CTRL, &vc);
}

void
add_menu_control(GuiCtrl *ctrl)
{
	int			i, n;
	struct v4l2_querymenu	qm;
	struct v4l2_input	vi;
	char			itemlabel[40];
	char			menulabel[40];

	strcpy(menulabel, ctrl->qc.name);
	strcat(menulabel, "...");
	ctrl->widget = XtVaCreateManagedWidget(
		"menubutton",
		menuButtonWidgetClass, ctrl->box,
		XtNlabel, menulabel,
		XtNmenuName, ctrl->qc.name,
		XtNborderWidth, 1,
		XtNorientation, XtorientHorizontal,
		XtNwidth, SLIDERLEN,
		NULL);
	ctrl->menu = XtVaCreatePopupShell(
		ctrl->qc.name,
		simpleMenuWidgetClass, ctrl->widget,
		NULL);

	for (i = n = 0; i < MAXMENU && i <= ctrl->qc.maximum; ++i)
	{
		if (ctrl->qc.id == VIDIN_ID)
		{
			vi.index = i;
			if (ioctl(vid, VIDIOC_ENUMINPUT, &vi) != 0)
				continue;
			strcpy(itemlabel, vi.name);
		}
		else
		{
			qm.id = ctrl->qc.id;
			qm.index = i;
			if (ioctl(vid, VIDIOC_QUERYMENU, &qm) != 0)
				break;
			strcpy(itemlabel, qm.name);
		}
		ctrl->menuentry[n].index = i;
		ctrl->menuentry[n].ctrl_id = ctrl->qc.id;
		ctrl->menuentry[n].menuitem = XtVaCreateManagedWidget(
			"item",
			smeBSBObjectClass, ctrl->menu,
			XtNlabel, itemlabel,
			XtNwidth, SLIDERLEN,
			NULL);
		XtAddCallback(ctrl->menuentry[n].menuitem, 
			      XtNcallback, menu_proc, &ctrl->menuentry[n]);
		++n;
	}
}


/*
 *  Action Button-type widgets for V4L2 BUTTON controls
 */

void
button_proc(Widget w, XtPointer client, XtPointer call)
{
	GuiCtrl			*ctrl = (GuiCtrl *)client;
	struct v4l2_control	vc;

	vc.id = ctrl->qc.id;
	vc.value = 0;
	ioctl(vid, VIDIOC_S_CTRL, &vc);
}

void
add_button_control(GuiCtrl *ctrl)
{
	ctrl->widget = XtVaCreateManagedWidget(
		ctrl->qc.name,
		commandWidgetClass, ctrl->box,
		XtNborderWidth, 2,
		XtNorientation, XtorientHorizontal,
		XtNwidth, SLIDERLEN,
		NULL);
	XtAddCallback(ctrl->widget, XtNcallback, button_proc, ctrl);
}


/*
 *  Top-level control-handling functions
 */

void
init_control(GuiCtrl *ctrl)
{
	if (ctrl->widget == 0)
		return;
	if (ctrl->qc.type == V4L2_CTRL_TYPE_INTEGER)
		init_scroll_control(ctrl);
	if (ctrl->qc.type == V4L2_CTRL_TYPE_BOOLEAN)
		init_bool_control(ctrl);
}

void
add_control(GuiCtrl *ctrl, Widget parent)
{
	ctrl->box =
		XtVaCreateManagedWidget("ctrlbox",
					boxWidgetClass, parent,
					XtNorientation, XtorientHorizontal,
					XtNhSpace, 2,
					XtNvSpace, 2,
					XtNborderWidth, 0,
					NULL);
	if (ctrl->qc.type == V4L2_CTRL_TYPE_INTEGER)
		add_scroll_control(ctrl);
	if (ctrl->qc.type == V4L2_CTRL_TYPE_BOOLEAN)
		add_bool_control(ctrl);
	if (ctrl->qc.type == V4L2_CTRL_TYPE_MENU)
		add_menu_control(ctrl);
	if (ctrl->qc.type == V4L2_CTRL_TYPE_BUTTON)
		add_button_control(ctrl);
	init_control(ctrl);
}

void
init_all_controls()
{
	int	i;
	for (i = 0; i < sizeof(ctrl)/sizeof(ctrl[0]); ++i)
		init_control(&ctrl[i]);
}

void
default_all_controls()
{
	struct v4l2_control	vc;
	int			i;

	for (i = 0; i < sizeof(ctrl)/sizeof(ctrl[0]); ++i)
	{
		if (ctrl[i].widget == 0)
			continue;
		if (ctrl[i].qc.type == V4L2_CTRL_TYPE_BUTTON)
			continue;
		vc.id = ctrl[i].qc.id;
		vc.value = ctrl[i].qc.default_value;
		ioctl(vid, VIDIOC_S_CTRL, &vc);
	}
	init_all_controls();
}

void
create_video_controls(Widget parent)
{
	int	i, id, err;

	app.w_videobox = XtVaCreateManagedWidget(
		"Video Controls",
		boxWidgetClass, parent,
		XtNorientation, XtorientVertical,
		XtNvSpace, 5,
		XtNborderWidth, 0,
		NULL);
	i = 0;

	/*  Video Input menu first (handled special)  */
	ctrl[i].qc.id = VIDIN_ID;
	ctrl[i].qc.type = V4L2_CTRL_TYPE_MENU;
	ctrl[i].qc.minimum = 0;
	ctrl[i].qc.maximum = 10;
	strcpy(ctrl[i].qc.name, "Video Input");
	add_control(&ctrl[i], app.w_videobox);
	++i;
	for (id = 0; id < 100; ++id)
	{
		ctrl[i].qc.id = V4L2_CID_BASE + id;
		err = ioctl(vid, VIDIOC_QUERYCTRL, &ctrl[i].qc);
		if (err && errno == EDOM)
			break;
		if (err == 0 && ctrl[i].qc.category == V4L2_CTRL_CAT_VIDEO)
		{
			add_control(&ctrl[i], app.w_videobox);
			++i;
		}
	}
	for (id = 0; id < 100; ++id)
	{
		ctrl[i].qc.id = V4L2_CID_PRIVATE_BASE + id;
		err = ioctl(vid, VIDIOC_QUERYCTRL, &ctrl[i].qc);
		if (err && errno == EDOM)
			break;
		if (err == 0 && ctrl[i].qc.category == V4L2_CTRL_CAT_VIDEO)
		{
			add_control(&ctrl[i], app.w_videobox);
			++i;
		}
	}
	app.w_videodefault = XtVaCreateManagedWidget(
		"Default All Controls",
		commandWidgetClass, app.w_videobox,
		XtNborderWidth, 2,
		XtNorientation, XtorientHorizontal,
		XtNwidth, SLIDERLEN,
		NULL);
	XtAddCallback(app.w_videodefault, 
		      XtNcallback, default_all_controls, 
		      NULL);


}



/*
 *  Main
 */
int
main(int argc, char *argv[])
{
	char			my_device[64];
	struct v4l2_capability	caps;
	char			device_label[96];


/*->	Put in the device node name */
	strcpy(my_device, MY_DEVICE);
	if (argc >= 2)
		strcpy(my_device, argv[1]);
	vid = open(my_device, O_NONCAP);
	if (vid < 0)
	{
		printf("No video device \"%s\"\n", my_device);
		return 1;
	}
	ioctl(vid, VIDIOC_QUERYCAP, &caps);


	app.w_toplevel = XtAppInitialize(
		&app.xtapp, "V4L2 Control Panel", NULL, 0,
		&argc, argv, NULL, NULL, 0);

	app.w_mainbox = XtVaCreateManagedWidget(
		"mainbox",
		boxWidgetClass, app.w_toplevel,
		XtNorientation, XtorientVertical,
		XtNhSpace, 0,
		XtNvSpace, 0,
		NULL);

	sprintf(device_label, "%s: %s", my_device, caps.name);
	app.w_device = XtVaCreateManagedWidget(
		"device",
		labelWidgetClass, app.w_mainbox,
		XtNlabel, device_label,
		XtNborderWidth, 0,
		NULL);

	app.w_menubox = XtVaCreateManagedWidget(
		"menubox",
		boxWidgetClass, app.w_mainbox,
		XtNorientation, XtorientHorizontal,
		XtNhSpace, 7,
		XtNvSpace, 2,
		XtNborderWidth, 0,
		NULL);

	app.w_quit = XtVaCreateManagedWidget(
		"Quit!",
		commandWidgetClass, app.w_menubox,
		XtNborderWidth, 2,
		NULL);
	XtAddCallback(app.w_quit, XtNcallback, quit, NULL);

	create_video_controls(app.w_mainbox);

	XtRealizeWidget(app.w_toplevel);
	XtAppMainLoop(app.xtapp);
	return 0;
}



