/*
 * for the Matrox Marvel G200 and Rainbow Runner-G series
 *
 * This module is an interface to the KS0127 video decoder chip.
 *
 * Copyright (C) 1999  Ryan Drake <stiletto@mediaone.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 *
 *****************************************************************************
 * Version History:
 * V1.0 Ryan Drake	   Initial version by Ryan Drake
 * V1.1 Gerard v.d. Horst  Added some debugoutput, reset the video-standard
 */

#ifndef __KERNEL__
#define __KERNEL__
#endif
#ifndef MODULE
#define MODULE
#endif

#include <linux/module.h>
#include <linux/delay.h>
#include <linux/errno.h>
#include <linux/kernel.h>
#include <linux/malloc.h>
#include <linux/i2c.h>
#include "ks0127.h"

#define dprintk     if (debug) printk

static int debug = 0; /* insmod parameter */

#if LINUX_VERSION_CODE >= 0x020100
MODULE_PARM(debug,"i");
#endif


#define KS0127_DEVNAME  "ks0127"


/* i2c identification */
#define I2C_KS0127_ADDON   0xD8
#define I2C_KS0127_ONBOARD 0xDA

/* ks0127 control registers */
#define KS_STAT     0x00
#define KS_CMDA     0x01
#define KS_CMDB     0x02
#define KS_CMDC     0x03
#define KS_CMDD     0x04
#define KS_HAVB     0x05
#define KS_HAVE     0x06
#define KS_HS1B     0x07
#define KS_HS1E     0x08
#define KS_HS2B     0x09
#define KS_HS2E     0x0a
#define KS_AGC      0x0b
#define KS_HXTRA    0x0c
#define KS_CMDE     0x0d
#define KS_PORTAB   0x0e
#define KS_LUMA     0x0f
#define KS_CON      0x10
#define KS_BRT      0x11
#define KS_CHROMA   0x12
#define KS_CHROMB   0x13
#define KS_DEMOD    0x14
#define KS_SAT      0x15
#define KS_HUE      0x16
#define KS_VERTIA   0x17
#define KS_VERTIB   0x18
#define KS_VERTIC   0x19
#define KS_HSCLL    0x1a
#define KS_HSCLH    0x1b
#define KS_VSCLL    0x1c
#define KS_VSCLH    0x1d
#define KS_OFMTA    0x1e
#define KS_OFMTB    0x1f
#define KS_VBICTL   0x20
#define KS_CCDAT2   0x21
#define KS_CCDAT1   0x22
#define KS_VBIL30   0x23
#define KS_VBIL74   0x24
#define KS_VBIL118  0x25
#define KS_VBIL1512 0x26
#define KS_TTFRAM   0x27
#define KS_TESTA    0x28
#define KS_UVOFFH   0x29
#define KS_UVOFFL   0x2a
#define KS_UGAIN    0x2b
#define KS_VGAIN    0x2c
#define KS_VAVB     0x2d
#define KS_VAVE     0x2e
#define KS_CTRACK   0x2f
#define KS_POLCTL   0x30
#define KS_REFCOD   0x31
#define KS_INVALY   0x32
#define KS_INVALU   0x33
#define KS_INVALV   0x34
#define KS_UNUSEY   0x35
#define KS_UNUSEU   0x36
#define KS_UNUSEV   0x37
#define KS_USRSAV   0x38
#define KS_USREAV   0x39
#define KS_SHS1A    0x3a
#define KS_SHS1B    0x3b
#define KS_SHS1C    0x3c
#define KS_CMDE2    0x3d
#define KS_VSDEL    0x3e
#define KS_CMDF     0x3f
#define KS_GAMMA0   0x40
#define KS_GAMMA1   0x41
#define KS_GAMMA2   0x42
#define KS_GAMMA3   0x43
#define KS_GAMMA4   0x44
#define KS_GAMMA5   0x45
#define KS_GAMMA6   0x46
#define KS_GAMMA7   0x47
#define KS_GAMMA8   0x48
#define KS_GAMMA9   0x49
#define KS_GAMMA10  0x4a
#define KS_GAMMA11  0x4b
#define KS_GAMMA12  0x4c
#define KS_GAMMA13  0x4d
#define KS_GAMMA14  0x4e
#define KS_GAMMA15  0x4f
#define KS_GAMMA16  0x50
#define KS_GAMMA17  0x51
#define KS_GAMMA18  0x52
#define KS_GAMMA19  0x53
#define KS_GAMMA20  0x54
#define KS_GAMMA21  0x55
#define KS_GAMMA22  0x56
#define KS_GAMMA23  0x57
#define KS_GAMMA24  0x58
#define KS_GAMMA25  0x59
#define KS_GAMMA26  0x5a
#define KS_GAMMA27  0x5b
#define KS_GAMMA28  0x5c
#define KS_GAMMA29  0x5d
#define KS_GAMMA30  0x5e
#define KS_GAMMA31  0x5f
#define KS_GAMMAD0  0x60
#define KS_GAMMAD1  0x61
#define KS_GAMMAD2  0x62
#define KS_GAMMAD3  0x63
#define KS_GAMMAD4  0x64
#define KS_GAMMAD5  0x65
#define KS_GAMMAD6  0x66
#define KS_GAMMAD7  0x67
#define KS_GAMMAD8  0x68
#define KS_GAMMAD9  0x69
#define KS_GAMMAD10 0x6a
#define KS_GAMMAD11 0x6b
#define KS_GAMMAD12 0x6c
#define KS_GAMMAD13 0x6d
#define KS_GAMMAD14 0x6e
#define KS_GAMMAD15 0x6f
#define KS_GAMMAD16 0x70
#define KS_GAMMAD17 0x71
#define KS_GAMMAD18 0x72
#define KS_GAMMAD19 0x73
#define KS_GAMMAD20 0x74
#define KS_GAMMAD21 0x75
#define KS_GAMMAD22 0x76
#define KS_GAMMAD23 0x77
#define KS_GAMMAD24 0x78
#define KS_GAMMAD25 0x79
#define KS_GAMMAD26 0x7a
#define KS_GAMMAD27 0x7b
#define KS_GAMMAD28 0x7c
#define KS_GAMMAD29 0x7d
#define KS_GAMMAD30 0x7e
#define KS_GAMMAD31 0x7f


/****************************************************************************
* mga_dev : represents one ks0127 chip.
****************************************************************************/
struct ks0127 {
	struct i2c_bus* bus;
        unsigned char addr;
	int		format_width;
	int		format_height;
	int		cap_width;
	int		cap_height;
};


/****************************************************************************
* raw register access : these routines directly interact with the ks0127's
*                       registers via the i2c
****************************************************************************/

#define I2C_SET(bus,ctrl,data)  (bus->i2c_setlines(bus,ctrl,data))
#define I2C_GET(bus)            (bus->i2c_getdataline(bus))

/* We need to manually read because of a bug in the KS0127 chip.
 *
 * An explanation from kayork@mail.utexas.edu:
 *
 * During I2C reads, the KS0127 only samples for a stop condition
 * during the place where the acknoledge bit should be. Any standard
 * I2C implementation (correctly) throws in another clock transition
 * at the 9th bit, and the KS0127 will not recognize the stop condition
 * and will continue to clock out data.
 *
 * So we have to do the read ourself.  Big deal.
 */
static u8 ks0127_read( struct ks0127* ks, u8 reg )
{
	char val = 0;
        int i;

	LOCK_FLAGS;
	LOCK_I2C_BUS( ks->bus );

        i2c_write( ks->bus, ks->addr, reg, 0, 0 );

        i2c_start( ks->bus );
        i2c_sendbyte( ks->bus, ks->addr | 1, 0 );
        I2C_SET(ks->bus,0,1);
        for (i=7; i>=0; i--) {
                I2C_SET(ks->bus,1,1);
                if (I2C_GET(ks->bus))
                        val |= (1<<i);
                I2C_SET(ks->bus,0,1);
        }
        /* no ack! */
        i2c_stop( ks->bus );
                        
        UNLOCK_I2C_BUS( ks->bus );
        return val;
}

static void ks0127_write( struct ks0127* ks, u8 reg, u8 val )
{
	LOCK_FLAGS;
	LOCK_I2C_BUS( ks->bus );
        i2c_write( ks->bus, ks->addr, reg, val, 1 );
        UNLOCK_I2C_BUS( ks->bus );
}

/* generic bit-twiddling */
static void ks0127_and_or( struct ks0127* ks, u8 reg, u8 and_v, u8 or_v )
{
        u8 val = ks0127_read( ks, reg );
        val = (val & and_v) | or_v;
        ks0127_write( ks, reg, val );
}

static int ks0127_probe( struct ks0127* ks )
{
        int ack;
	LOCK_FLAGS;

	LOCK_I2C_BUS( ks->bus );
	ack = i2c_sendbyte( ks->bus, ks->addr, 2000 );
	UNLOCK_I2C_BUS( ks->bus );
        return ack;
}


/****************************************************************************
* ks0127 private api
****************************************************************************/
static void ks0127_reset( struct ks0127* ks )
{
        int i;
        static u8 table[] =
        {
/* 00 */        0x97,0x0c,0x02,0x42,0x01,0x0c,0x04,0x0c,
/* 08 */        0x00,0x00,0x00,0x53,0x00,0x00,0x0f,0x02,
/* 10 */        0x00,0x00,0x2a,0x00,0x80,0x00,0x00,0x60,
#ifdef	NEVER
/* 18 */        0x14,0x48,0x01,0x00,0x00,0x00,0x20,0x00,
#else
/* 18 */        0x10,0x4b,0x01,0x00,0x00,0x00,0x20,0x00,
#endif
/* 20 */        0xfd,0x00,0x00,0xa8,0xaa,0x2a,0x00,0x00,
#ifdef	NEVER
/* 28 */        0x09,0x00,0x00,0x00,0x00,0x07,0x00,0x00,
#else
/* 28 */        0x09,0x00,0x00,0x00,0x00,0x13,0x7c,0x00,
#endif
/* 30 */        0x41,0xa4,0x00,0x00,0x00,0x00,0x00,0x00,
/* 38 */        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x06
        };

	dprintk( "ks0127: reset\n" );
        udelay(1000);

        /* initialize all registers to known values (except STAT which is ro) */
        for( i=1; i<sizeof(table); i++ )
                ks0127_write( ks, i, table[i] );

//        ks0127_and_or( ks, KS_OFMTA,  0xef, 0x20 );
//        ks0127_and_or( ks, KS_PORTAB, 0xff, 0x0f );
//        ks0127_and_or( ks, KS_CMDD,   0xff, 0x01 );
}


/****************************************************************************
* ks0127 i2c driver interface
****************************************************************************/
static int ks0127_attach( struct i2c_device* device )
{
	struct ks0127* ks;

	device->data = ks = kmalloc( sizeof( *ks ), GFP_KERNEL );
	if( ks == NULL )
		return -ENOMEM;
	memset( ks, 0, sizeof( *ks ) );

        strcpy( device->name, KS0127_DEVNAME );
	ks->bus = device->bus;
	ks->addr = device->addr;

        /* probe the device to determine who it is */
        if( !ks0127_probe( ks ) )
        {
                /* nobody home */
                kfree( ks );
                return -1;
        }

        /* reset the device */
        ks0127_reset( ks );
#ifdef	NEVER
	printk(KERN_INFO "ks0127: attach: %s video decoder\n", ks->addr==I2C_KS0127_ADDON?"addon":"on-board" );
#endif
        MOD_INC_USE_COUNT;
	return 0;
}

static int ks0127_detach(struct i2c_device* device)
{
	struct ks0127* ks = (struct ks0127*)device->data;

	ks0127_reset( ks );

	kfree( ks );

	dprintk( "ks0127: detach\n" );
	MOD_DEC_USE_COUNT;
	return 0;
}

void ks0127_getcrop(struct ks0127 *ks, int *xstart, int *xend, int *ystart, int *yend)
{
	*xstart = ((ks0127_read(ks, KS_HXTRA) & 0xe0) << 3) + 
		ks0127_read(ks, KS_HAVB);

	*xend = ((ks0127_read(ks, KS_HXTRA) & 0x1c) << 6) + 
		ks0127_read(ks, KS_HAVE);

	if ((*xstart) & 0x400) 
		*xstart |= ~0x3ff;

	if ((*xend) & 0x400) 
		*xend |= ~0x3ff;

	*ystart = ks0127_read(ks, KS_VAVB) >> 2;
	*yend = ks0127_read(ks, KS_VAVE);
dprintk("ks0127: ystart=%d yend=%d\n", *ystart, *yend);

	*ystart = (*ystart - 2) * 2;
	*yend = (*yend - 2) * 2;
}

static int ks0127_command( struct i2c_device* device, unsigned int cmd, void* arg )
{
	struct ks0127*	ks = (struct ks0127*)device->data;
	int		*iarg = (int*)arg;
	int		size;
	int		xstart;
	int		ystart;
	int		xend;
	int		yend;
	int		hav;
	int		vav;

        if( !ks )
                return -ENODEV;

	dprintk( "ks0127: command %x\n", cmd );

	switch (cmd)
        {
        case KS0127_RESET:
	        dprintk( "ks0127: command KS0127_RESET\n" );
                ks0127_reset( ks );
                break;

        case KS0127_SET_INPUT:
                switch( *iarg )
                {
                case KS_INPUT_COMPOSITE:
	            dprintk( "ks0127: command KS0127_SET_INPUT: INPUT_COMPOSITE\n" );
                    ks0127_and_or( ks, KS_CMDA,   0xfc, 0x00 ); /* autodetect 50/60 Hz */
                    ks0127_and_or( ks, KS_CMDB,   0xf0, 0x02 ); /* set input line = AY2 */
                    ks0127_and_or( ks, KS_CMDC,   0x70, 0x0a ); /* non-freerunning mode */
                    ks0127_and_or( ks, KS_CMDD,   0xe7, 0x00 ); /* analog input */
                    ks0127_and_or( ks, KS_CTRACK, 0xc7, 0x00 ); /* enable chroma demodulation */
                    break;

                case KS_INPUT_SVIDEO:
	            dprintk( "ks0127: command KS0127_SET_INPUT: INPUT_SVIDEO\n" );
                    ks0127_and_or( ks, KS_CMDA,   0xfc, 0x00 ); /* autodetect 50/60 Hz */
                    ks0127_and_or( ks, KS_CMDB,   0xf0, 0x08 ); /* set input line = AY0-AC0 */
                    ks0127_and_or( ks, KS_CMDC,   0x70, 0x0a ); /* non-freerunning mode */
                    ks0127_and_or( ks, KS_CMDD,   0xe7, 0x00 ); /* analog input */
                    ks0127_and_or( ks, KS_CTRACK, 0xc7, 0x00 ); /* enable chroma demodulation */
                    break;

                case KS_INPUT_TUNER:
	            dprintk( "ks0127: command KS0127_SET_INPUT: INPUT_TUNER\n" );
                    ks0127_and_or( ks, KS_CMDA,   0xfc, 0x00 ); /* autodetect 50/60 Hz */
                    ks0127_and_or( ks, KS_CMDB,   0xf0, 0x01 ); /* set input line = AY1 */
                    ks0127_and_or( ks, KS_CMDC,   0x70, 0x0a ); /* non-freerunning mode */
                    ks0127_and_or( ks, KS_CMDD,   0xe7, 0x00 ); /* analog input */
                    ks0127_and_or( ks, KS_CTRACK, 0xc7, 0x00 ); /* enable chroma demodulation */
                    break;

                case KS_INPUT_YUV656_60HZ:
	            dprintk( "ks0127: command KS0127_SET_INPUT: INPUT_YUV656_60HZ\n" );
                    ks0127_and_or( ks, KS_CMDA,   0xfc, 0x03 ); /* force 60 Hz */
                    ks0127_and_or( ks, KS_CMDB,   0xf0, 0x08 ); /* set input line = AY0-AC0 */
                    ks0127_and_or( ks, KS_CMDC,   0x70, 0x87 ); /* freerunning mode */
                    ks0127_and_or( ks, KS_CMDD,   0xe7, 0x08 ); /* digital input */
                    ks0127_and_or( ks, KS_CTRACK, 0xc7, 0x20 ); /* disable chroma demodulation */
                    break;

                case KS_INPUT_YUV656_50HZ:
	            dprintk( "ks0127: command KS0127_SET_INPUT: INPUT_YUV656_50HZ\n" );
                    ks0127_and_or( ks, KS_CMDA,   0xfc, 0x02 ); /* force 50 Hz */
                    ks0127_and_or( ks, KS_CMDB,   0xf0, 0x08 ); /* set input line = AY0-AC0 */
                    ks0127_and_or( ks, KS_CMDC,   0x70, 0x87 ); /* freerunning mode */
                    ks0127_and_or( ks, KS_CMDD,   0xe7, 0x08 ); /* digital input */
                    ks0127_and_or( ks, KS_CTRACK, 0xc7, 0x20 ); /* disable chroma demodulation */
                    break;
                }
                break;

        case KS0127_SET_OUTPUT:
                switch( *iarg )
                {
                case KS_OUTPUT_YUV656E:
	            dprintk( "ks0127: command KS0127_SET_OUTPUT: OUTPUT_YUV656E\n" );
                    ks0127_and_or( ks, KS_OFMTA, 0xc0, 0x33);
                    break;

                case KS_OUTPUT_EXV:
	            dprintk( "ks0127: command KS0127_SET_OUTPUT: OUTPUT_EXV\n" );
                    ks0127_and_or( ks, KS_OFMTA, 0xc0, 0x39 );
                    break;
                }
                break;

        case KS0127_SET_STANDARD:
                /* Set to automatic SECAM/Fsc mode */
                ks0127_and_or( ks, KS_DEMOD, 0xf0, 0x00 );

                switch( *iarg )
                {
                case KS_STD_NTSC_M:
	            dprintk( "ks0127: command KS0127_SET_STANDARD: NTSC_M\n" ); 
                    ks0127_and_or( ks, KS_CHROMA, 0x9f, 0x00 );
		    ks->format_height = 240;
		    ks->format_width = 704;
                    break;

                case KS_STD_NTSC_N:
	            dprintk( "ks0127: command KS0127_SET_STANDARD: NTSC_N\n" );
                    ks0127_and_or( ks, KS_CHROMA, 0x9f, 0x00 );
		    ks->format_height = 240;
		    ks->format_width = 704;
                    break;

                case KS_STD_PAL_N:
	            dprintk( "ks0127: command KS0127_SET_STANDARD: PAL_N\n" );
                    ks0127_and_or( ks, KS_CHROMA, 0x9f, 0x60 );
		    ks->format_height = 290;
		    ks->format_width = 704;
                    break;

                case KS_STD_PAL_M:
	            dprintk( "ks0127: command KS0127_SET_STANDARD: PAL_M\n" );
                    ks0127_and_or( ks, KS_CHROMA, 0x9f, 0x60 );
		    ks->format_height = 290;
		    ks->format_width = 704;
                    break;

                case KS_STD_SECAM:
	            dprintk( "ks0127: command KS0127_SET_STANDARD: SECAM\n" );
		    ks->format_height = 290;
		    ks->format_width = 704;

                    /* set to secam autodetection */
                    ks0127_and_or( ks, KS_CHROMA, 0xdf, 0x20 );
                    ks0127_and_or( ks, KS_DEMOD, 0xf0, 0x00 );
                    udelay( 100000 );

                    /* did it autodetect? */
                    if( ks0127_read( ks, KS_DEMOD ) & 0x40 )
                        break;

                    /* force to secam mode */
                    ks0127_and_or( ks, KS_DEMOD, 0xf0, 0x0f );
                    break;
                }
                break;

        case KS0127_SET_BRIGHTNESS:
                ks0127_write( ks, KS_BRT, *iarg );
                break;

        case KS0127_SET_CONTRAST:
                ks0127_write( ks, KS_CON, *iarg );
                break;

        case KS0127_SET_HUE:
                ks0127_write( ks, KS_HUE, *iarg );
                break;

        case KS0127_SET_SATURATION:
                ks0127_write( ks, KS_SAT, *iarg );
		if (*iarg == 0) 
			ks0127_and_or( ks, KS_UVOFFH, 0xb0, 0x00);
		 else 
			ks0127_and_or( ks, KS_UVOFFL, 0x00, 0x40);

                break;

        case KS0127_SET_AGC_MODE:
                switch( *iarg )
                {
                case KS_AGC_NORMAL:
                    ks0127_and_or( ks, KS_CMDB, 0x4f, 0x00 );
                    break;

                case KS_AGC_MODE1:
                    ks0127_and_or( ks, KS_CMDB, 0x4f, 0x80 );
                    break;

                case KS_AGC_MANUAL:
                    ks0127_and_or( ks, KS_CMDB, 0xcf, 0x10 );
                    break;

                case KS_AGC_AUTO:
                    ks0127_and_or( ks, KS_CMDB, 0xdf, 0x20 );
                    break;
                }
                break;

        case KS0127_SET_AGC:
                ks0127_write( ks, KS_AGC, *iarg );
                break;
                
        case KS0127_SET_CHROMA_MODE:
                switch( *iarg )
                {
                case KS_CHROMA_AUTO:
                    ks0127_and_or( ks, KS_CHROMA, 0xfc, 0x00 );
                    break;

                case KS_CHROMA_ON:
                    ks0127_and_or( ks, KS_CHROMA, 0xfc, 0x02 );
                    break;

                case KS_CHROMA_OFF:
                    ks0127_and_or( ks, KS_CHROMA, 0xfc, 0x03 );
                    break;
                }
                break;

        case KS0127_SET_PIXCLK_MODE:
                switch( *iarg )
                {
                case KS_PIXCLK_CCIR601:
                    ks0127_and_or( ks, KS_CMDA, 0xfb, 0x04 );
                    break;

                case KS_PIXCLK_SQUARE:
                    ks0127_and_or( ks, KS_CMDA, 0xfb, 0x00 );
                    break;
                }
                break;

        case KS0127_SET_GAMMA_MODE:
                switch( *iarg )
                {
                case KS_GAMMA_OFF:
                    ks0127_and_or( ks, KS_OFMTA, 0x3f, 0x00 );
                    break;

                case KS_GAMMA_Y:
                    ks0127_and_or( ks, KS_OFMTA, 0x3f, 0x40 );
                    break;
                
                case KS_GAMMA_UV:
                    ks0127_and_or( ks, KS_OFMTA, 0x3f, 0x80 );
                    break;

                case KS_GAMMA_YUV:
                    ks0127_and_or( ks, KS_OFMTA, 0x3f, 0xc0 );
                    break;
                }
                break;

        case KS0127_SET_UGAIN:
                ks0127_write( ks, KS_UGAIN, *iarg );
                break;

        case KS0127_SET_VGAIN:
                ks0127_write( ks, KS_VGAIN, *iarg );
                break;

        case KS0127_SET_INVALY:
                ks0127_write( ks, KS_INVALY, *iarg );
                break;

        case KS0127_SET_INVALU:
                ks0127_write( ks, KS_INVALU, *iarg );
                break;

        case KS0127_SET_INVALV:
                ks0127_write( ks, KS_INVALV, *iarg );
                break;

        case KS0127_SET_UNUSEY:
                ks0127_write( ks, KS_UNUSEY, *iarg );
                break;

        case KS0127_SET_UNUSEU:
                ks0127_write( ks, KS_UNUSEU, *iarg );
                break;

        case KS0127_SET_UNUSEV:
                ks0127_write( ks, KS_UNUSEV, *iarg );
                break;

        case KS0127_SET_VSALIGN_MODE:
                switch( *iarg )
                {
                case KS_VSALIGN0:
                    ks0127_and_or( ks, KS_CMDB, 0xbf, 0x00 );
                    break;

                case KS_VSALIGN1:
                    ks0127_and_or( ks, KS_CMDB, 0xbf, 0x40 );
                    break;
                }
                break;

        case KS0127_SET_OUTPUT_MODE:
                switch( *iarg )
                {
                case KS_OUTPUT_NORMAL:
                    ks0127_and_or( ks, KS_CMDE,  0x7f, 0x00 );
                    ks0127_and_or( ks, KS_OFMTA, 0xcf, 0x00 );
                    break;

                case KS_OUTPUT_HIZ:
                    ks0127_and_or( ks, KS_CMDE,  0x7f, 0x80 );
                    ks0127_and_or( ks, KS_OFMTA, 0xcf, 0x00 );
                    break;

                case KS_OUTPUT_SYNC_HIZ:
                    ks0127_and_or( ks, KS_CMDE,  0x7f, 0x80 );
                    ks0127_and_or( ks, KS_OFMTA, 0xcf, 0x10 );
                    break;

                case KS_OUTPUT_SYNC_CLK_HIZ:
                    ks0127_and_or( ks, KS_CMDE,  0x7f, 0x80 );
                    ks0127_and_or( ks, KS_OFMTA, 0xcf, 0x20 );
                    break;
                }
                break;

	case KS0127_SET_WIDTH:
		ks0127_getcrop( ks, &xstart, &xend, &ystart, &yend);
		dprintk("ks0127: xstart=%d xend=%d ystart=%d, yend=%d\n",
			xstart, xend, ystart, yend);

		hav = 720 - (xstart + xend);
		vav = yend - ystart;
		dprintk("ks0127: hav = %d vav=%d\n", hav, vav);

		size = (*iarg * 0x7fff) / hav;
		ks->cap_width = *iarg;
		dprintk("ks0127: width size = %d program = 0x%x\n", *iarg, size);
		ks0127_and_or( ks, KS_HSCLL, 0x01, size << 1);
		ks0127_write( ks, KS_HSCLH, size >> 7);


#ifdef	NEVER
		/*
		 * This little hack can for samsung note was
		 * scaler fix -- turn off fifi for at least 1 line
		 */
		for(i=0; i < 10; i++) 
			ks0127_and_or( ks, 0x0d, ~0x03, 0);
		ks0127_and_or( ks, 0x0d, ~0x03, 1);
#endif
		

		if (ks->cap_width <= 384) {
			ks0127_and_or( ks, KS_VERTIA, ~0x08, 0x00);
		} else {

			ks0127_and_or( ks, KS_VERTIA, ~0x08, 0x08);
		}

		break;

	case KS0127_SET_HEIGHT:
#ifdef	NEVER
		ks0127_getcrop( ks, &xstart, &xend, &ystart, &yend);
		dprintk("ks0127: xstart=%d xend=%d ystart=%d, yend=%d\n",
			xstart, xend, ystart, yend);

		hav = 720 - (xstart + xend);
		vav = yend - ystart;
		dprintk("ks0127: hav = %d vav=%d\n", hav, vav);

		size = (*iarg * 0x3fff) / vav;
		ks->cap_height = *iarg;
		dprintk("ks0127: height size = %d program = 0x%x\n", *iarg, size);

		h1 = ks0127_read( ks, KS_HSCLL);
		h2 = ks0127_read( ks, KS_HSCLH);

		ks0127_and_or( ks, KS_VSCLL, 0x03, 0xfc);
		ks0127_write( ks, KS_VSCLH, 0xff);
		ks0127_and_or( ks, KS_HSCLL, 0x01, 0xfe);
		ks0127_write( ks, KS_HSCLH, 0);

		ks0127_and_or( ks, KS_VSCLL, 0x03, size << 2);
		ks0127_write( ks, KS_VSCLH, size >> 6);

		ks0127_and_or( ks, KS_VERTIB, ~0x06, 0x00);

		ks0127_and_or( ks, KS_HSCLL, 0x01, h1);
		ks0127_write( ks, KS_HSCLH, h2);

		
#ifdef	NEVER
		if (ks->cap_height == ks->format_height) {
			ks0127_and_or( ks, KS_VERTIB, ~0x06, 0x04);
		} else {
			ks0127_and_or( ks, KS_VERTIB, ~0x06, 0x00);
		}

		if (ks->cap_height <= ks->format_height/2) {
			ks0127_and_or( ks, KS_VERTIC, ~0x04, 0x04);
		} else {
			ks0127_and_or( ks, KS_VERTIC, ~0x04, 0x00);
		}
#endif
#endif
		break;


        case KS0127_GET_STATUS:
                *iarg = ks0127_read( ks, KS_STAT );
                break;

	default:
		return -EINVAL;
	}

	return 0;
}


struct i2c_driver i2c_driver_ks0127 = {
        KS0127_DEVNAME,
	I2C_DRIVERID_KS0127,
	I2C_KS0127_ADDON,
        I2C_KS0127_ONBOARD,
	ks0127_attach,
	ks0127_detach,
	ks0127_command
};


/****************************************************************************
* linux kernel module api
****************************************************************************/
#ifdef MODULE
int init_module(void)
#else
int ks0127_init(void)
#endif
{
	i2c_register_driver( &i2c_driver_ks0127 );
	return 0;
}

#ifdef MODULE
void cleanup_module(void)
{
	i2c_unregister_driver( &i2c_driver_ks0127 );
}
#endif
