/*
 * Video Capture Driver ( Video for Linux 1/2 )
 * for the Matrox Marvel G200 and Rainbow Runner-G series
 *
 * This module is an interface to the MAVEN tv-out chip.
 *
 * Copyright (C) 1999  Ryan Drake <stiletto@mediaone.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#ifndef __KERNEL__
#define __KERNEL__
#endif
#ifndef MODULE
#define MODULE
#endif

#include <linux/module.h>
#include <linux/delay.h>
#include <linux/errno.h>
#include <linux/kernel.h>
#include <linux/malloc.h>
#include <linux/i2c.h>
/* #include "maven.h"  - no such file yet */

#define dprintk     if (debug) printk

static int debug = 0; /* insmod parameter */

#if LINUX_VERSION_CODE >= 0x020100
MODULE_PARM(debug,"i");
#endif


#define MAVEN_DEVNAME  "maven"


/* i2c identification */
#define I2C_MAVEN 0x36

/* maven control registers */
/* TODO */


/****************************************************************************
* mga_dev : represents one maven chip.
****************************************************************************/
struct maven {
	struct i2c_bus* bus;
        unsigned char addr;
};


/****************************************************************************
* raw register access : these routines directly interact with the maven's
*                       registers via the i2c
****************************************************************************/

static u8 maven_read( struct maven* mv, u8 reg )
{
	char val = 0;
	LOCK_FLAGS;
	LOCK_I2C_BUS( mv->bus );
        i2c_write( mv->bus, mv->addr, reg, 0, 0 );
        val = i2c_read( mv->bus, mv->addr );
        UNLOCK_I2C_BUS( mv->bus );
        return val;
}

static void maven_write( struct maven* mv, u8 reg, u8 val )
{
	LOCK_FLAGS;
	LOCK_I2C_BUS( mv->bus );
        i2c_write( mv->bus, mv->addr, reg, val, 1 );
        UNLOCK_I2C_BUS( mv->bus );
}

/* generic bit-twiddling */
static void maven_and_or( struct maven* mv, u8 reg, u8 and_v, u8 or_v )
{
        u8 val = maven_read( mv, reg );
        val = (val & and_v) | or_v;
        maven_write( mv, reg, val );
}

static int maven_probe( struct maven* mv )
{
        int ack;
	LOCK_FLAGS;

	LOCK_I2C_BUS( mv->bus );
	ack = i2c_sendbyte( mv->bus, mv->addr, 2000 );
	UNLOCK_I2C_BUS( mv->bus );

        return ack;
}


/****************************************************************************
* maven private api
****************************************************************************/
static void maven_reset( struct maven* mv )
{
}


/****************************************************************************
* maven i2c driver interface
****************************************************************************/
static int maven_attach( struct i2c_device* device )
{
	struct maven* mv;

	device->data = mv = kmalloc( sizeof( *mv ), GFP_KERNEL );
	if( mv == NULL )
		return -ENOMEM;
	memset( mv, 0, sizeof( *mv ) );

        strcpy( device->name, MAVEN_DEVNAME );
	mv->bus = device->bus;
	mv->addr = device->addr;

        /* probe the device to determine who it is */
        if( !maven_probe( mv ) )
        {
                /* nobody home */
                kfree( mv );
                return -1;
        }

        /* reset the device */
        maven_reset( mv );

	printk(KERN_INFO "maven: attach: tv-out\n" );
        MOD_INC_USE_COUNT;
	return 0;
}

static int maven_detach(struct i2c_device* device)
{
	struct maven* mv = (struct maven*)device->data;

	maven_reset( mv );

	kfree( mv );

	dprintk( "maven: detach\n" );
	MOD_DEC_USE_COUNT;
	return 0;
}

static int maven_command( struct i2c_device* device, unsigned int cmd, void* arg )
{
	struct maven* mv = (struct maven*)device->data;
/*	int* iarg = (int*)arg;
*/
        if( !mv )
                return -ENODEV;

	dprintk( "maven: command %x\n", cmd );

	switch (cmd)
        {
        /* TODO */
	default:
		return -EINVAL;
	}

	return 0;
}


struct i2c_driver i2c_driver_maven = {
        MAVEN_DEVNAME,
	I2C_DRIVERID_MAVEN,
	I2C_MAVEN,
        I2C_MAVEN,
	maven_attach,
	maven_detach,
	maven_command
};


/****************************************************************************
* linux kernel module api
****************************************************************************/
#ifdef MODULE
int init_module(void)
#else
int maven_init(void)
#endif
{
	i2c_register_driver( &i2c_driver_maven );
	return 0;
}

#ifdef MODULE
void cleanup_module(void)
{
	i2c_unregister_driver( &i2c_driver_maven );
}
#endif
