
/*
 *
 * Copyright (C) 1999-2000  Thomas Mirlacher
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * The author may be reached as <dent@linuxvideo.org>
 *
 *------------------------------------------------------------
 *
 * May 29th 2000: Dave Sainty <dave@dtsp.co.nz>
 *	NetBSD fixes
 *	correced under/overflows (strncat)
 *
 * Nov 1th 2000: Brian Graham <s3065808@student.anu.edu.au>
 *	Proxy support
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <netdb.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <sys/stat.h>

#include "dvddb.h"
#include "dvddb_conf.h"

struct server_struct {
	const char *name;
	int port;
};

#ifndef INADDR_NONE
#define INADDR_NONE 0xFFFFFFFF
#endif

static FILE *_dvddbOpen (const char *id, const char *file_mode);
static char *_proxy_address (void);
static int _proxy_port (void);

int proxy_enable = 1;  // to disable proxy, set to 0

/**
 * Open a connection to the DVDDB server.
 **/
	
int dvddbNetConnect (void)
{
	int sock;
	struct sockaddr_in sin;
	struct hostent *host;
	struct server_struct server;
	struct server_struct proxy;

	proxy.name = _proxy_address ();
	proxy.port = _proxy_port ();

	if (proxy_enable && proxy.name) {
		server.name = proxy.name;
		server.port = proxy.port;
	} else {
		server.name = DVDDB_HOST;
		server.port = DVDDB_HOST_PORT;
	}

	sin.sin_family = AF_INET;
	sin.sin_port = htons(server.port);

	if ((sin.sin_addr.s_addr = inet_addr(server.name)) == INADDR_NONE) {
		if (!(host = gethostbyname (server.name))) {
			LOG (LOG_ERROR, "gethostbyname %s", strerror (errno));
			return -1;
		}

		memcpy (&sin.sin_addr, host->h_addr, host->h_length);
	}

	if ((sock = socket (AF_INET, SOCK_STREAM, 0)) < 0) {
		LOG (LOG_ERROR, "socket %s", strerror (errno));
		return -1;
	}

	if (connect(sock, (struct sockaddr *) &sin, sizeof(sin)) < 0) {
		LOG (LOG_ERROR, "connect %s", strerror (errno));
		return -1;
	}

	return sock;
}


/**
 * Close connection previously established to the DVDDB server.
 **/

int dvddbNetDisconnect (int sock)
{
	return close (sock);
}


/**
 * Put a new entry in the database (on server) 
 **/

int dvddbNetPut (int sock, int fd, char *id)
{
	char buf[BUFSIZ];
	int len;
	int ret;

// get file size
	len = (int) lseek (fd, 0, SEEK_END);
	lseek (fd, 0, SEEK_SET);
	
// form HTTP header
	//ret = snprintf (buf, BUFSIZ, "POST /%s?%s HTTP/1.0\nContent-length: %d\n\n",
	ret = snprintf (buf, BUFSIZ, "POST http://%s:%i/%s?%s HTTP/1.0\nContent-length: %d\n\n",
			DVDDB_HOST,
			DVDDB_HOST_PORT,
			DVDDB_HOST_DB,
			id,
			len);

	if (ret < 0 || ret >= BUFSIZ) {
		LOG (LOG_ERROR, "%s", strerror (errno));
		return -1;
	}

// write HTTP header
	write (sock, buf, strlen (buf));

// write data
	while ((len = read (fd, buf, BUFSIZ)) > 0) {
#ifdef DEBUG
	{
		int i;

		for (i=0; i<len; i++)
			fprintf (stderr, "%c", buf[i]);
	}
#endif
		if (write (sock, buf, len) < 0) {
			LOG (LOG_ERROR, "%s", strerror (errno));
			return -1;
		}
	}
	
	return 0;
}


/**
 * Get an entry from the server database (and cache it locally)
 * DENT: clean up here ... so this is a place where most likely 
 *	a seqfault appears	
 **/

int dvddbNetGet (int sock, char *id)
{
	int http_hdr_found = 0;
	int db_hdr_found = 0;
	char buf[BUFSIZ];
	int len;
	int ret;
	int i;
	FILE *file = NULL;


// request specific ID from host
	ret = snprintf (buf, BUFSIZ, "GET http://%s:%i/%s?%s HTTP/1.0\n\n",
			DVDDB_HOST,
			DVDDB_HOST_PORT,
			DVDDB_HOST_DB,
			id);

	if (ret < 0 || ret >= BUFSIZ) {
		LOG (LOG_ERROR, "output truncated");
		return -1;
	}

	write (sock, buf, strlen (buf));

// and now, wait for reply, drop HTTP header and put rest int
//	local cache (file)
 
	while ((len = read (sock, buf, BUFSIZ)) > 0) {
		i = 0;

		if (!http_hdr_found) {
			for (i=0; (i<len-1) && !http_hdr_found; i++) {
				if (	(buf[i  ] == '\n') &&
			    		(buf[i+1] == '\r')) {

					http_hdr_found = 1;	
					i+=2;
				}
			}
		}

// parsing dvddb_hdr
		if (http_hdr_found && !db_hdr_found) {
			char const *token;
			token = strtok (buf+i, ":");

			while (!db_hdr_found && token) {
				if (!strcasecmp (token, "version")) {
					int version;

					version = atoi (strtok (NULL, "\n\r"));
				
					if (version > DVDDB_VERSION) {
						LOG (LOG_ERROR, "server version (v%d) is greater than client version (v%d)\n - please upgrade", version, DVDDB_VERSION);
						fclose (file);
						return -1;
					}	
				} else if (!strcasecmp (token, "data")) {
					db_hdr_found = 1;
					token = strtok (NULL, "\n\r");
					if (!strcasecmp (token, " no"))
						return -1;
					token = strtok (NULL, "\n\r");
					i = token - &buf[0];

					if ((file = _dvddbOpen (id, "w")) == NULL)
					return -1;

					break;
				}

				token = strtok (NULL, ":");
			}
		} 

		if (db_hdr_found) {
			fwrite (&buf[i], len-i, 1, file);	
#ifdef DEBUG
			for (; i<len; i++)
				fprintf (stderr, "%c", buf[i]);
#endif
		}
	}

	fclose (file);

	return 0;
}


/**
 *
 **/

#ifndef MAXPATHLEN
#define MAXPATHLEN 1024
#endif

static FILE *_dvddbOpen (char const *id, char const *file_mode)
{
	FILE *file;
	char file_name[MAXPATHLEN];
	char file_name_tmp[MAXPATHLEN];
        struct stat stat_buffer;
     	int ret;
 
// check for oms user-config dir 
	{
		const char *home = getenv ("HOME");
		ret = snprintf (file_name, sizeof(file_name), "%s/.oms", home ? home : "");

		if (ret < 0 || ret >= sizeof(file_name))
			goto err;
	}

        if (stat (file_name, &stat_buffer))
		mkdir (file_name, S_IRUSR | S_IWUSR | S_IXUSR);
  
// check for dvddb dir 
	ret = snprintf (file_name_tmp, MAXPATHLEN, "%s/dvddb", file_name);

	if (ret < 0 || ret >= MAXPATHLEN)
		goto err;
   
        if (stat (file_name_tmp, &stat_buffer))
		mkdir (file_name_tmp, S_IRUSR | S_IWUSR | S_IXUSR);
 
// prepare to read/write from/to the id file   
	ret = snprintf (file_name, MAXPATHLEN, "%s/%s", file_name_tmp, id);

	if (ret < 0 || ret >= MAXPATHLEN)
		goto err;

        LOG (LOG_DEBUG, "_dvddbOpen %s for %s ...",
		file_name,
		*file_mode == 'r' ? "reading" : "writing");

	if (!(file = fopen (file_name, file_mode))) {
		LOG (LOG_INFO, "cannot open DVDDB cache file (%s)\n", file_name);
                return NULL;
        }

	LOG (LOG_DEBUG, "OK");

        return file;

err:
	LOG (LOG_ERROR, "error creating DVDDB filename-string\n");
	return NULL;
}


/**
 *
 **/

FILE *dvddbOpen (char *id)
{
	return _dvddbOpen (id, "r");
}

/**
 *
 **/

int dvddbClose (FILE *file)
{
	if (!file)
		return -1;

	return fclose (file);
}


/**
 *
 **/

char *dvddbRead (FILE *file)
{
	static char *buf = NULL;

	if (!file)
		return NULL;

	if (!buf) {
		if (!(buf = malloc (BUFSIZ))) {
			LOG (LOG_ERROR, "memory squeeze");
			return NULL;
		}
	}

	if (fgets (buf, BUFSIZ, file)) {
		size_t len = strlen (buf);

		if (len > 0 && buf[len - 1] == '\n')
			buf[len - 1] = '\0';      // delete newline

		return buf;
	}

	if (buf)
		free (buf);

	return NULL;
}


/**
 * DENT: move this into the UDF code - if christian agrees
 *	SOMEONE SHOULD CHECK IF THIS ID IS UNIQUE
 **/



#define DVD_BLKSIZE ((off_t) 0x800)
#define OFF_ID_BLK ((off_t) 0x20)
#define OFF_ID_BYTE ((off_t) 0x49)

int dvddbGetID (char *device, char *id)
{
	int fd;
	char buf[DVD_BLKSIZE];
	int ret = -1;

	if ((fd = open (device, O_RDONLY)) < 0) {
		LOG (LOG_ERROR, "unable to open device (%s)", device);
		return ret;
	}

	if (lseek (fd, DVD_BLKSIZE * OFF_ID_BLK, SEEK_SET) < 0) {
		LOG (LOG_ERROR, "lseek %s", strerror (errno));
		goto err;
	}

	if (read (fd, buf, DVD_BLKSIZE) < DVD_BLKSIZE) {
		LOG (LOG_ERROR, "short read on ID");
		goto err;
	}

	memcpy (id, &buf[OFF_ID_BYTE], 8);
	id[8] = '\0';

	ret = 0;
err:
        close (fd);

        return ret;
}


static int _proxy_port (void)
{
	char *proxy = getenv ("http_proxy");

	if (!proxy) {			// try upper case
		if (!(proxy = getenv ("HTTP_PROXY")))
			return PROXY_DEFAULT_PORT;
	}

	if (!strncasecmp("http://", proxy, 7)) {
		int i;

		for (i=7; proxy[i]; i++) {
			if (proxy[i] == ':')
				return atoi (&proxy[i+1]);
		}
	}

	return PROXY_DEFAULT_PORT;
}


static char *_proxy_address (void)
{
	char *proxy = getenv ("http_proxy");
	char *our_proxy;
	int i;

	if (!proxy) {		// try upper case
		if (!(proxy = getenv ("HTTP_PROXY")))
			return NULL;
	}

	if (!(our_proxy = malloc (strlen (proxy) * sizeof (char))))
		return NULL;

	if (!strncasecmp("http://", proxy, 7)) {
		for (i=7; proxy[i] && proxy[i] != '/' && proxy[i] != ':'; i++) {
         		our_proxy[i-7] = proxy[i];
		}
      		our_proxy[i-7] = '\0';
		return our_proxy;
	} else { // assume http_proxy is of the form address or address:port
		for (i=0; proxy[i] != ':' && proxy[i] != '/' && proxy[i]; i++) {
			our_proxy[i] = proxy[i];
		}
		our_proxy[i] = '\0';
		return our_proxy;
	}

	free (our_proxy);
	return NULL;
}
