/*
 *
 * Copyright (C) 2000 Yuqing Deng <Yuqing_Deng@brown.edu>
 *
 * navigation packet structure from vobdump by Eric Smith <eric@brouhaha.com>
 * 
 * program organization and bit stream buffering stolen from extract_ac3.c
 * by Aaron Holtzman <aholtzma@engr.uvir.ca>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *
 *------------------------------------------------------------  
 */


//#define SEEK_PIPES

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/stat.h>
#include <sys/mman.h>
#include <inttypes.h>

#include <bswap.h>
#include "nav.h"

#define BUFSIZE 2048 /* needs to be as big as biggest header */
#define CHECK_ZERO(x) if ((x)) printf ("*** %s not zero (0x%04x)\n", #x, x)

extern void ifoPrintVMOP(uint8_t *opcode);
extern void decode_user_op (uint32_t u_op);

static int vobf;
static uint8_t buf[BUFSIZE];
static uint8_t *cur_pos;
static uint8_t *end_pos;
static uint32_t nbtns;
static uint32_t btn_grp;


void print_pci_gi (uint8_t *pci)
{
        int i;

        nav_pci_gi *gi = (nav_pci_gi*) pci;
	printf ("current block:   0x%08x\n", be2me_32(gi->nv_pck_lbn));
	printf ("vobu_cat:        0x%04x\n", be2me_16(gi->vobu_cat));

	if (gi->reserved)
		printf ("reserved:        0x%04x\n", gi->reserved);
	printf ("user operations: 0x%08x\n", be2me_32(gi->vobu_uop_ctl));
	decode_user_op(be2me_32(gi->vobu_uop_ctl));
	printf ("vobu_s_ptm:      0x%08x\n", be2me_32(gi->vobu_s_ptm));
	printf ("vobu_e_ptm:      0x%08x\n", be2me_32(gi->vobu_e_ptm));
	printf ("vobu_se_e_ptm:   0x%08x\n", be2me_32(gi->vobu_se_e_ptm));
	printf ("e_eltm:          0x%08x\n", be2me_32(gi->e_eltm));

	printf ("vobu_isrc:     \"");
	for (i = 0; i < 32; i++) {
	  uint8_t c;
	  if ((c = gi->vobu_isrc[i]) >= ' ' && (c <= '~'))
	    printf ("%c", c);
	  else
	    printf (".");
	}
	printf ("\"\n");

	for (i = 0; i < 9; i++) {
	  printf ("nsml_agl_c%d_dsta:  0x%08x\n", i, be2me_32(gi->nsml_agli_dsta[i]));
	}
}

void print_pci_hli (uint8_t *pci)
{
        nav_pci_hli_gi *hli = (nav_pci_hli_gi*) pci;

	CHECK_ZERO (hli->reserved);
	CHECK_ZERO (hli->reserved0);
	printf ("hli_ss:          0x%01x\n", hli->hli_ss);

	btn_grp = hli->btngr_ns;
	nbtns = hli->btn_ns;

	printf ("hli_s_ptm:       0x%02x%02x%02x%02x\n",
		hli->hli_s_ptm[0], hli->hli_s_ptm[1],
		hli->hli_s_ptm[2], hli->hli_s_ptm[3]);
	printf ("hli_e_ptm:       0x%02x%02x%02x%02x\n",
		hli->hli_e_ptm[0], hli->hli_e_ptm[1],
		hli->hli_e_ptm[2], hli->hli_e_ptm[3]);
	printf ("btn_se_e_ptm:    0x%02x%02x%02x%02x\n",
		hli->btn_se_e_ptm[0], hli->btn_se_e_ptm[1],
		hli->btn_se_e_ptm[2], hli->btn_se_e_ptm[3]);
	
	CHECK_ZERO (hli->reserved1);
	printf ("button groups:   0x%01x\n", hli->btngr_ns);
	CHECK_ZERO (hli->reserved2);
	printf ("btngr0_dsp_ty:   0x%01x\n", hli->btngr0_dsp_ty);
	CHECK_ZERO (hli->reserved3);
	printf ("btngr1_dsp_ty:   0x%01x\n", hli->btngr1_dsp_ty);
	CHECK_ZERO (hli->reserved4);
	printf ("btngr2_dsp_ty:   0x%01x\n", hli->btngr2_dsp_ty);
	
	printf ("btn_ofn:         0x%02x\n", hli->btn_ofn);
	CHECK_ZERO (hli->reserved5);
	printf ("btn_ns:          0x%02x\n", hli->btn_ns);

	CHECK_ZERO (hli->reserved6);
	printf ("nsl_btn_ns:      0x%02x\n", hli->nsl_btn_ns);

	CHECK_ZERO (hli->reserved7);
	CHECK_ZERO (hli->reserved8);

	printf ("fosl_btnn:       0x%02x\n", hli->fosl_btnn);

	CHECK_ZERO (hli->reserved9);
	printf ("foac_btnn:       0x%02x\n", hli->foac_btnn);
	printf ("btn_coli_0 sl: 0x%02x%02x%02x%02x ac: 0x%02x%02x%02x%02x\n",
		hli->btn_coli_0_sl[0], hli->btn_coli_0_sl[1],
		hli->btn_coli_0_sl[2], hli->btn_coli_0_sl[3],
		hli->btn_coli_0_ac[0], hli->btn_coli_0_ac[1],
		hli->btn_coli_0_ac[2], hli->btn_coli_0_ac[3]);
	printf ("btn_coli_1 sl: 0x%02x%02x%02x%02x ac: 0x%02x%02x%02x%02x\n",
		hli->btn_coli_1_sl[0], hli->btn_coli_1_sl[1],
		hli->btn_coli_1_sl[2], hli->btn_coli_1_sl[3],
		hli->btn_coli_1_ac[0], hli->btn_coli_1_ac[1],
		hli->btn_coli_1_ac[2], hli->btn_coli_1_ac[3]);
	printf ("btn_coli_2 sl: 0x%02x%02x%02x%02x ac: 0x%02x%02x%02x%02x\n",
		hli->btn_coli_2_sl[0], hli->btn_coli_2_sl[1],
		hli->btn_coli_2_sl[2], hli->btn_coli_2_sl[3],
		hli->btn_coli_2_ac[0], hli->btn_coli_2_ac[1],
		hli->btn_coli_2_ac[2], hli->btn_coli_2_ac[3]);
       
}

	
void print_pci_btns (uint8_t* pci, uint32_t nbtns, uint32_t btn_gprs)
{
	int i, j;

	for (i = 0; i < btn_gprs; i++) {
		for (j = 0; j < 36 / btn_gprs; j++, pci += 18) {
			if (j < nbtns) {
				nav_pci_btnit *btnit = (nav_pci_btnit *) pci;
				uint32_t x_start, y_start, x_end, y_end;
				x_start = btnit->x_start_f6;
				x_start <<= 4; x_start += btnit->x_start_l4;
				y_start = btnit->y_start_f6;
				y_start <<= 4; y_start += btnit->y_start_l4;
				x_end = btnit->x_end_f2;
				x_end <<= 8; x_end += btnit->x_end_l8;
				y_end = btnit->y_end_f2;
				y_end <<= 8; y_end += btnit->y_end_l8;
	
				printf ("group %d btni %d:  ", i, j);
				printf ("btn_coln %d, auto_action_mode %d\n",
					btnit->btn_coln, btnit->auto_action_mode); 
				printf ("coords   (%d, %d) - (%d, %d)\n",
					x_start, y_start, x_end, y_end);                                
				printf ("up %d, ", btnit->up);  
				printf ("down %d, ", btnit->down); 
				printf ("left %d, ", btnit->left);
				printf ("right %d\n", btnit->right);
	
				ifoPrintVMOP(&(btnit->btn_command[0]));
				printf ("\n");	     
			}
		}
	}
}
       
                                                        
void print_pci(uint8_t* pci)
{
	uint8_t *cur = pci;

	if (*cur++ == 0x00) {
		printf ("pci packet\n");
	} else {
		printf ("not a pci packet\n");
	}

	print_pci_gi(cur);
	cur += sizeof(nav_pci_gi);
	print_pci_hli(cur);
	cur += 46;
	print_pci_btns(cur, nbtns, btn_grp);
}


void print_dsi_gi(uint8_t *gi)
{
	nav_dsi_gi *dsi_gi = (nav_dsi_gi *) gi;
	
	printf ("nv_pck_scr     0x%08x\n", be2me_32(dsi_gi->nv_pck_scr));
	printf ("nv_pck_lbn     0x%08x\n", be2me_32(dsi_gi->nv_pck_lbn));
	printf ("vobu_ea        0x%08x\n", be2me_32(dsi_gi->vobu_ea));
	printf ("vobu_1stref_ea 0x%08x\n", be2me_32(dsi_gi->vobu_1stref_ea));
	printf ("vobu_2ndref_ea 0x%08x\n", be2me_32(dsi_gi->vobu_2ndref_ea));
	printf ("vobu_3rdref_ea 0x%08x\n", be2me_32(dsi_gi->vobu_3rdref_ea));
	printf ("vob id:        0x%04x\n", be2me_16(dsi_gi->vobu_vob_idn));
	printf ("reserved:      0x%02x\n", dsi_gi->reserved);
	printf ("cell id:       0x%02x\n", dsi_gi->vobu_c_idn);
	printf ("c_eltm:        0x%04x\n", be2me_32(dsi_gi->c_eltm));
}


void print_dsi(uint8_t* dsi)
{
        uint8_t *cur = dsi;
	if (*cur++ == 0x01) {
		printf ("dsi pack\n");
	} else {
		printf ("not a dsi packet\n");
	}
	print_dsi_gi(cur);
	cur += sizeof(nav_dsi_gi);
	//print_dsi_sml_pbi(cur);
	// cur += 148;
	// print_dsi_sml_anli(cur);
	// cur += 54;
	// print_vobu_sri(cur);
	// cur += 168;
	// print_synci(cur); // 144 bytes
}

void file_init(char file_name[])
{
	if(file_name[0] == '-' && file_name[1] == '\0') {
		vobf = STDIN_FILENO;
	} else if ((vobf = open(file_name, O_RDONLY)) < 0) {
		printf ("File not found\n");
		exit(1);
	}
	cur_pos = buf;
	end_pos = buf;
}

inline void increment_position(long x)
{
	if(cur_pos + x <= end_pos) {
		cur_pos += x;
		if(cur_pos == end_pos) {
			cur_pos = buf;
			end_pos = buf;
		}
	} else {
		long size = 0;
		x -= (long)(end_pos - cur_pos);
#ifdef SEEK_PIPES
		if(lseek(vobf, x, SEEK_CUR) < 0) {
			printf ("Error: unexpected end of stream\n");
			exit(1);
		}
#else
		while(x)
		{
		   size = (x > BUFSIZE) ? BUFSIZE : x;
		   if(read(vobf, buf, size) < size)
		   {
		      printf ("Error: unexpected end of stream\n");
		      exit(1);
		   }
                   x-=size;
		}
#endif
		cur_pos = buf;
                end_pos = buf;
	}
}


inline static void load_next_bytes(long count)
{
	if(cur_pos + count <= end_pos)
		return;

	if(cur_pos + count > buf + BUFSIZE - 1 ) {
		printf ("No buffer space to read %ld bytes\n", count);
		exit(1);
	}

	count -= (long)(end_pos - cur_pos);

	if(read(vobf, end_pos, count) < count) {
		printf ("Error: unexpected end of stream\n");
		exit(1);
	}

	end_pos += count;
}


inline int next_24_bits(long x)
{
	load_next_bytes(3);

	if (cur_pos[0] != ((x >> 16) & 0xff))
		return 0;
	if (cur_pos[1] != ((x >>  8) & 0xff))
		return 0;
	if (cur_pos[2] != ((x      ) & 0xff))
		return 0;

	return 1;
}


inline int next_32_bits(long x)
{
	load_next_bytes(4);

	if (cur_pos[0] != ((x >> 24) & 0xff))
		return 0;
	if (cur_pos[1] != ((x >> 16) & 0xff))
		return 0;
	if (cur_pos[2] != ((x >>  8) & 0xff))
		return 0;
	if (cur_pos[3] != ((x      ) & 0xff))
		return 0;

	return 1;
}


void read_write_next_bytes(long count, int outfd)
{
	long size;
	size = (long)(end_pos - cur_pos);

	if(size > count) {
		write(outfd, cur_pos, count);
		cur_pos +=count;

		if(cur_pos == end_pos) {
			cur_pos = buf;
			end_pos = buf;
		}
		return;
	} else if (size > 0) {
		write(outfd, cur_pos, size);
	}

	while(count) {
		size = (count > BUFSIZE) ? BUFSIZE : count;

		if(read(vobf, buf, size) < size ||
			write(outfd, buf, size) < size) {
			printf ("Error: unexpected end of stream\n");
		}
		count -= size;
	}
	cur_pos = buf;
	end_pos = buf;
}


void parse_pes(void)
{
	unsigned long data_length;
	unsigned long header_length;

	load_next_bytes(9);

	//The header length is the PES_header_data_length byte plus 6 for the packet
	//start code and packet size, 3 for the PES_header_data_length and two
	//misc bytes, and finally 4 bytes for the mystery AC3 packet tag 
	header_length = cur_pos[8] + 6 + 3;
	data_length =(cur_pos[4]<<8) + cur_pos[5];

	if(cur_pos[3] == 0xbf) {
		load_next_bytes(data_length - 3);

		// pci packet
		printf ("start of pci curpos[] = %02x%02x%02x%02x\n",
			cur_pos[0],cur_pos[1],cur_pos[2],cur_pos[3]);
		printf ("data_length = %lx\n", data_length);
		print_pci(&cur_pos[6]);
		
		increment_position(data_length + 6);

		// dsi packet
		load_next_bytes(9);

		data_length = (cur_pos[4] << 8) + cur_pos[5];
		printf ("start of dsi curpos[] = %02x%02x%02x%02x\n",
			cur_pos[0],cur_pos[1],cur_pos[2],cur_pos[3]);
		printf ("data_length = %lx\n", data_length);
		load_next_bytes(data_length - 3);
		print_dsi(&cur_pos[6]);
		
		increment_position(data_length + 6);
	} else {
	  increment_position(data_length + 6);
	}
}


void parse_pack(void)
{
	unsigned long skip_length;

	// Deal with the pack header 
	// The first 13 bytes are junk. The fourteenth byte 
	// contains the number of stuff bytes 
	load_next_bytes(14);
	skip_length = cur_pos[13] & 0x7;
	increment_position(14 + skip_length);

	// Deal with the system header if it exists 
	if(next_32_bits(0x000001bb))
	{
	// Bytes 5 and 6 contain the length of the header minus 6
                load_next_bytes(6);
		skip_length = (cur_pos[4] << 8) +  cur_pos[5];
		increment_position(6 + skip_length);
	}

	while(next_24_bits(0x000001) && !next_32_bits(0x000001ba)) {
		parse_pes ();
	}
}


int main(int argc, char *argv[])
{

	if (argc < 2) {
		printf ("usage: %s mpeg_stream [track number]\n", argv[0]);
		exit(1);
	}

	file_init(argv[1]);

	if (!next_32_bits(0x000001ba)) {
		printf ("Non-program streams not handled - exiting\n\n");
		exit(1);
	}

	do {
		parse_pack();
	} while (next_32_bits(0x000001ba));

	printf ("curpos[] = %x%x%x%x\n",cur_pos[0],cur_pos[1],cur_pos[2],cur_pos[3]);

	if(!next_32_bits(0x000001b9)) {
		printf ("Error: expected end of stream code\n");
		exit(1);
	}

	if(vobf != STDIN_FILENO) close(vobf);
	return 0;
}		

