/*
 * Copyright (C) 2000 Yoann Vandoorselaere
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Authors:   Yoann Vandoorselaere <yoann@mandrakesoft.com>
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <dlfcn.h>
#include <assert.h>

#include <oms/oms.h>
#include <oms/log.h>



static config_t *cfg = NULL;
static char *audio_plugin = NULL;
static char *video_plugin = NULL;



static void output_audio_update(void)
{
	const char *plugin, *device;

	plugin = config_get(cfg, NULL, "output_audio_plugin");
        assert(plugin);
        
	if (!audio_plugin || (plugin && strcmp(plugin, audio_plugin) != 0)) {
                free(audio_plugin);
                audio_plugin = strdup(plugin);

		device = config_get(cfg, NULL, "output_audio_device");
		if (!device)
			device = "/dev/dsp";

		omsCloseOutputAudio();
		omsOpenOutputAudio(plugin, plugin, device);
	}
}



static void output_video_update(void)
{
	const char *plugin, *device;

	plugin = config_get(cfg, NULL, "output_video_plugin");
        assert(plugin);
        
        if (!video_plugin || (plugin && strcmp(plugin, video_plugin) != 0)) {
                free(video_plugin);
                video_plugin = strdup(plugin);

                device = config_get (cfg, NULL, "output_video_device");
                if ( ! device )
                        device = "/dev/null";

		omsCloseOutputVideo();
		omsOpenOutputVideo(plugin, plugin, device);
	}
}



/*
 * Wake up plugin that are active and have a config cb
 * registered... 
 */
static void call_plugin_cb (void)
{
	int i;
	plugin_t *plugin;

	for (i = 0; i < MAX_PLUGIN_ID; i++) {
		plugin = plugin_get_active_ops(i);
		if ( plugin && plugin->config )
			plugin->config();
	}
}



void omsConfigApply (void)
{
	output_video_update();
	output_audio_update();
	call_plugin_cb();
}



void omsConfigOk (void)
{
	omsConfigApply();
	config_write(cfg);
}



static void _create_default_config (void)
{
	const char *ptr;

	/*
	 * Create basic and needed entry.
	 */
	if (!(ptr = config_get (cfg, NULL, "output_audio_plugin")))
		config_set (cfg, NULL, "output_audio_plugin", "sys");

	if (!(ptr = config_get (cfg, NULL, "output_audio_device")))
		config_set (cfg, NULL, "output_audio_device", "/dev/dsp");

	if (!(ptr = config_get (cfg, NULL, "output_video_plugin")))
		config_set (cfg, NULL, "output_video_plugin", "x11");

	if (!(ptr = config_get (cfg, NULL, "output_video_device")))
		config_set (cfg, NULL, "output_video_device", "/dev/null");
}



config_t *omsConfigOpen(void)
{
        if (! cfg ) {
		char filename[256];

		snprintf(filename, sizeof(filename),
                         "%s/.oms/config", getenv("HOME"));

                cfg = config_open(filename);
                if ( ! cfg ) {
			LOG(LOG_ERROR, "couldn't open %s.", filename);
			exit(1);
		}

                _create_default_config();
	}
        
	return cfg;
}
