//#define USE_LIBLTDL
#define USE_DEPEND 1

/*
 *
 * Copyright (C) 2000  Thomas Mirlacher
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * The author may be reached as <dent@linuxvideo.org>
 *
 *------------------------------------------------------------
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <ctype.h>
#include <dlfcn.h>
#include <dirent.h>
#include <inttypes.h>
#include <assert.h>

#include <oms/oms.h>
#include <oms/plugin.h>
#include <oms/fourcc.h>
#include <oms/plugin/id2name.h>
#include <oms/plugin/input.h>
#include <oms/plugin/decaps.h>

static plugin_entry_t *plugin_list[MAX_PLUGIN_ID];
static plugin_entry_t *plugin_active_list[MAX_PLUGIN_ID];

static char *plugin_fourcc_in[MAX_PLUGIN_ID];
static char *plugin_fourcc_out[MAX_PLUGIN_ID];

#if USE_DEPEND
// depend is a circular linked list linking together the dependent plugins
static plugin_entry_t *depend_first;
static plugin_entry_t *depend_prev;
#endif

static lt_dlhandle plugin_handle[MAX_PLUGIN_ID];
static int register_time = 1;

plugin_entry_t *plugin_find_dep (uint32_t plugin_id, uint32_t fourcc,
			       plugin_entry_t * depend);

#if BYTE_ORDER == BIG_ENDIAN
uint32_t string2FOURCC (const char *string)
{
	int i;
	uint32_t fourcc = 0;

	for (i = 0; i < 4 && string[i]; i++)
		fourcc = (fourcc << 8) | string[i];

	for (; i < 4; i++)
		fourcc = (fourcc << 8) | ' ';

	return fourcc;
}
#else
uint32_t string2FOURCC (const char *string)
{
	int i;
	uint32_t fourcc = 0;

	for (i = 0; i < 4 && string[i]; i++)
		fourcc |= string[i] << (8 * i);

	for (; i < 4; i++)
		fourcc |= ' ' << (8 * i);

	return fourcc;
}
#endif


#ifdef DEBUG
static void plugin_print_all(void) 
{
        int plugin_type;
        
        fprintf (stderr, "---------------------\n");
	
	for (plugin_type = 0; plugin_type < MAX_PLUGIN_ID; plugin_type++) {
		plugin_entry_t *entry = plugin_get_registered (plugin_type);
                
		while (entry) {
			fprintf (stderr, "%d: [%s]\t[%s]\t%s\n",
				 entry->plugin_id,
				 entry->fourcc_in ? (char *) entry->
				 fourcc_in : "[null]",
				 entry->fourcc_out ? (char *) entry->
				 fourcc_out : "[null]", entry->whoami);

			entry = entry->next;
		}
	}

	fprintf (stderr, "---------------------\n");
}
#endif



int plugin_mgr_init (void)
{
        const char *plugin_dir;
        config_t *cfg;

        lt_dlinit ();

        cfg = omsConfigOpen();
        assert(cfg);
        
        plugin_dir = config_get(cfg, NULL, "plugin_dir");
        if ( ! plugin_dir )
                plugin_dir = PLUGIN_DIR;
        
	plugin_mgr_scandir(plugin_dir);
        
#ifdef DEBUG
        plugin_print_all();
#endif
	return 0;
}

void plugin_unregister (plugin_entry_t * entry)
{
	if (!entry)
		return;

	if (entry->whoami)
		free (entry->whoami);

	if (entry->fourcc_in)
		free (entry->fourcc_in);

	if (entry->fourcc_out)
		free (entry->fourcc_out);

	if (entry->xml_config)
		free (entry->xml_config);

	free (entry);
}

int plugin_mgr_exit (void)
{
	int plugin_type;
	plugin_entry_t *entry,
	*entry_next;

	for (plugin_type = 0; plugin_type < MAX_PLUGIN_ID; plugin_type++) {	// loop plugin types
	        if (plugin_fourcc_in[plugin_type])
		        free (plugin_fourcc_in[plugin_type]);
	        if (plugin_fourcc_out[plugin_type])
                        free(plugin_fourcc_out[plugin_type]);
		entry = plugin_get_registered (plugin_type);

		while (entry) {
			entry_next = entry->next;
			plugin_unregister (entry);
			entry = entry_next;
		}

		plugin_list[plugin_type] = NULL;
	}

	return lt_dlexit ();
}

static void _register_in_order (plugin_entry_t * entry, uint plugin_id)
{
	entry->next = plugin_list[plugin_id];
	plugin_list[plugin_id] = entry;
}

static void _register_sorted (plugin_entry_t * entry, uint plugin_id)
{
	plugin_entry_t *tmp_entry = plugin_get_registered (plugin_id);
	plugin_entry_t *prev_entry = tmp_entry;

	entry->magic_len = ((plugin_decaps_t *) entry->ops)->magic->len;
	while (tmp_entry) {
		if (tmp_entry->magic_len <= entry->magic_len)
			break;

		prev_entry = tmp_entry;
		tmp_entry = tmp_entry->next;
	}

	if (!plugin_get_registered (plugin_id)) {
		_register_in_order (entry, plugin_id);
	} else {
		entry->next = prev_entry->next;
		prev_entry->next = entry;
	}
}

static uint32_t *_plugin_string2fourcc (const char *_string)
{
	char *substr;
	uint32_t *fourcc = NULL;
	int num = 0;
	char *string,
	*ptr;

	if (!_string)
		return NULL;

	ptr = string = strdup (_string);

	for (num = 2; (substr = strsep (&string, ";")); num++) {
		if (!
		    (fourcc =
		     (uint32_t *) realloc (fourcc,
					   num * sizeof (uint32_t)))) {
			LOG (LOG_ERROR, "memory squeeze");
			free (ptr);
			free (fourcc);
			return NULL;
		}
		fourcc[num - 2] = string2FOURCC (substr);
		fourcc[num - 1] = 0;
	}

	free (ptr);

	return fourcc;
}

static int _fourcc_cmp (uint32_t * fourcc_list, uint32_t * fourcc)
{
	uint32_t *ptr = fourcc_list;

	if (!fourcc_list && !fourcc)
		return 0;

	if (!fourcc_list || !fourcc)
		return -1;

	while (*ptr) {
		if (*fourcc == *ptr)
			return 0;

		ptr++;
	}

	return -1;
}

static void _plugin_register_at_init (char *whoami, uint plugin_id,
				      char *fourcc_in, char *fourcc_out,
				      char *xml_config, void *plugin_data)
{
	plugin_entry_t *entry = calloc (1, sizeof (plugin_entry_t));

#ifdef DEBUG
	printf ("registering plugin %s ", whoami);
	printf ("\tplugin_id: %x", plugin_id);
	printf ("\tfourcc_in: %s\n", fourcc_in ? fourcc_in : "[null]");
	printf ("\tfourcc_out: %s\n", fourcc_out ? fourcc_out : "[null]");
#endif

	entry->whoami = whoami ? strdup (whoami) : NULL;
	entry->plugin_id = plugin_id;
	entry->fourcc_in = _plugin_string2fourcc (fourcc_in);
	entry->fourcc_out = _plugin_string2fourcc (fourcc_out);
	entry->xml_config = xml_config ? strdup (xml_config) : NULL;

#if USE_DEPEND
	if (!depend_first)
		depend_first = entry;
	else
		entry->depend = depend_prev;

	depend_prev = entry;
#endif

	entry->ops = plugin_data;

	if (plugin_id == PLUGIN_ID_DECAPS)
		_register_sorted (entry, plugin_id);
	else
		_register_in_order (entry, plugin_id);
}

static int _plugin_register_after_init (char *whoami, uint plugin_id,
					uint32_t * fourcc_in,
					uint32_t * fourcc_out,
					void *plugin_data)
{
	plugin_entry_t *entry = plugin_get_registered (plugin_id);

	for (; entry; entry = entry->next) {
		if (!strcmp (entry->whoami, whoami)) {
			entry->ops = plugin_data;
			return 0;
		}
	}

	return -1;
}

int pluginRegister (char *whoami, uint plugin_id, char *fourcc_in,
		    char *fourcc_out, char *xml_config, void *plugin_data)
{
	int ret;
	uint32_t *in,
	*out;

	if (register_time) {
		_plugin_register_at_init (whoami, plugin_id, fourcc_in,
					  fourcc_out, xml_config,
					  plugin_data);
		return 0;
	}

	in = _plugin_string2fourcc (fourcc_in);
	out = _plugin_string2fourcc (fourcc_out);

	ret =
	    _plugin_register_after_init (whoami, plugin_id, in, out,
					 plugin_data);

	free (in);
	free (out);

	return ret;
}

void *plugin_get_active_ops (uint plugin_id)
{
	if (!plugin_active_list[plugin_id])
		return NULL;

	return plugin_active_list[plugin_id]->ops;
}

plugin_entry_t *plugin_get_active (uint plugin_id)
{
	return plugin_active_list[plugin_id];
}

int plugin_set_active (uint plugin_id, plugin_entry_t * entry)
{
	LOG (LOG_DEBUG, "setting plugin_id %s active (%d)", entry->whoami,
	     plugin_id);

	plugin_active_list[plugin_id] = entry;
	return 0;
}

char *plugin_get_fourcc_in (uint plugin_id)
{
        return plugin_fourcc_in[plugin_id];
}

int plugin_set_fourcc_in (uint plugin_id, const char *fourcc)
{
        LOG (LOG_DEBUG, "setting plugin_id %d fourcc_in %s)", plugin_id, fourcc);
        if (plugin_fourcc_in[plugin_id]) free(plugin_fourcc_in[plugin_id]);
        plugin_fourcc_in[plugin_id] = fourcc ? strdup(fourcc) : NULL;
	
        return 0;
}

char *plugin_get_fourcc_out (uint plugin_id)
{
        return plugin_fourcc_out[plugin_id];
}

int plugin_set_fourcc_out (uint plugin_id, const char *fourcc)
{
        LOG (LOG_DEBUG, "setting plugin_id %d fourcc_out %s)", plugin_id, fourcc);
        if (plugin_fourcc_out[plugin_id]) free(plugin_fourcc_out[plugin_id]);
        plugin_fourcc_out[plugin_id] = fourcc ? strdup(fourcc) : NULL;
	
        return 0;
}

plugin_entry_t *plugin_get_registered (uint plugin_id)
{
	return plugin_list[plugin_id];
}

plugin_entry_t *plugin_find (uint32_t plugin_id, uint32_t * fourcc_in,
			    uint32_t * fourcc_out)
{
	plugin_entry_t *entry;

	entry = plugin_get_registered (plugin_id);

	for (; entry; entry = entry->next) {
#if USE_DEPEND
		if (plugin_id != PLUGIN_ID_OUTPUT_VIDEO
		    && plugin_id != PLUGIN_ID_OUTPUT_AUDIO && entry->depend)	// only find non-dependent plugins here
			continue;
#endif

		if (_fourcc_cmp (entry->fourcc_in, fourcc_in))
			continue;

		if (_fourcc_cmp (entry->fourcc_out, fourcc_out))
			continue;

		LOG (LOG_INFO, "plugin %s matched %s, %s : %s, %s",
		     entry->whoami, (char *) entry->fourcc_in,
		     (char *) entry->fourcc_out, (char *) fourcc_in,
		     (char *) fourcc_out);
		return entry;
	}

	return NULL;
}

plugin_entry_t *plugin_find_dep (uint32_t plugin_id, uint32_t fourcc_in,
			       plugin_entry_t * depend)
{
	plugin_entry_t *entry = depend;

	while (entry) {
		if (entry->plugin_id == plugin_id) {
			if (!_fourcc_cmp (entry->fourcc_in, &fourcc_in))
				return entry;
		}

		entry = entry->depend;

		if (entry == depend)	// are we running in circles?
			// we have to but 1 time is enough
			break;
	}

	return NULL;
}

static lt_dlhandle _plugin_load (char *path)
{
	int (*init) (void *);
	lt_dlhandle handle;

	if (!(handle = lt_dlopenext (path))) {
		LOG (LOG_ERROR,
		     "PLUGIN: %s doesn't seem to be installed (%s)", path,
		     lt_dlerror ());
		exit (-1);
		return NULL;
	}

	if (!(init = lt_dlsym (handle, "plugin_init"))) {
		LOG (LOG_ERROR,
		     "%s unable to find plugin-init-function (%s)", path,
		     lt_dlerror ());
		return NULL;
	}

	if ((*init) (path)) {
		LOG (LOG_ERROR, "%s unable to initialize (%s)\n",
		     path, lt_dlerror ());
		return NULL;
	}

	return handle;
}

//DENT: please don't change functions, so they are useless - like this one ..
//      fourcc_in and out is REALLY useless here.

int plugin_load_path (uint32_t plugin_id, char *path, uint32_t * fourcc_in,
		    uint32_t * fourcc_out)
{
	plugin_entry_t *entry;

	LOG (LOG_INFO, "loading plugin: (%s) %s", pluginid2name[plugin_id],
	     path);

	if (plugin_id >= MAX_PLUGIN_ID) {
		LOG (LOG_ERROR, "unknown plugin type");
		return -1;
	}

	if (!(entry = plugin_find (plugin_id, fourcc_in, fourcc_out))) {
		LOG (LOG_DEBUG, "cannot find plugin %x %s", plugin_id,
		     path);
		return -1;
	}

	if (!(plugin_handle[plugin_id] = _plugin_load (entry->whoami)))
		return -1;

	if (plugin_set_active (plugin_id, entry) < 0)
		return -1;

	LOG (LOG_DEBUG, "plugin %s loaded", path);

	return 0;
}

/**
 * return OPS(function pointers) of loaded plugin
 **/

void *pluginLoad (uint32_t plugin_id, const char *fourcc_in,
		  const char *fourcc_out)
{
	plugin_entry_t *entry = NULL;

#if USE_DEPEND
	plugin_entry_t *output_plugin = NULL;
#endif
	lt_dlhandle handle;
	uint32_t *in,
	*out;

	LOG (LOG_INFO, "loading plugin: (%s) %s,  %s",
	     pluginid2name[plugin_id], (char *) fourcc_in,
	     (char *) fourcc_out);

	if (plugin_id >= MAX_PLUGIN_ID) {
		LOG (LOG_ERROR, "unknown plugin type");
		return NULL;
	}

	in = _plugin_string2fourcc (fourcc_in);
	out = _plugin_string2fourcc (fourcc_out);

#if USE_DEPEND
	if (plugin_id == PLUGIN_ID_CODEC_AUDIO)
		output_plugin = plugin_get_active (PLUGIN_ID_OUTPUT_AUDIO);
	else if (plugin_id == PLUGIN_ID_CODEC_VIDEO || plugin_id == PLUGIN_ID_CODEC_SPU)
		output_plugin = plugin_get_active (PLUGIN_ID_OUTPUT_VIDEO);

	if (output_plugin && output_plugin->depend) {
		LOG (LOG_INFO,
		     "found dependency - looking for the right device");
// look for plugin which depends on this one (looping through the depend list)
		entry =
		    plugin_find_dep (plugin_id, *in, output_plugin->depend);

		if (!entry)
			LOG (LOG_ERROR, "didn't find a dependent plugin");
	}

	if (!entry)
		entry = plugin_find (plugin_id, in, out);
#else
	entry = plugin_find (plugin_id, in, out);
#endif

	free (in);
	free (out);

	if (!entry)
		return NULL;

	if (!(handle = _plugin_load (entry->whoami)))
		return NULL;

	plugin_handle[plugin_id] = handle;

	if (plugin_set_active (plugin_id, entry) < 0)
		return NULL;

        if (plugin_set_fourcc_in (plugin_id, fourcc_in) < 0)
	        return NULL;

        if (plugin_set_fourcc_out (plugin_id, fourcc_out) < 0)
	        return NULL;

	return plugin_get_active_ops (plugin_id);
}

int plugin_unload (uint plugin_id)
{
	if (plugin_id >= MAX_PLUGIN_ID) {
		LOG (LOG_ERROR, "unknown plugin type %x", plugin_id);
		return -1;
	}

	if (plugin_handle[plugin_id]) {
		plugin_t *plugin = (plugin_t *) plugin_get_active_ops (plugin_id);
		plugin->close (plugin);

		LOG (LOG_INFO, "unloading plugin (%s)",
		     pluginid2name[plugin_id]);
		if (lt_dlclose (plugin_handle[plugin_id]) < 0) {
			LOG (LOG_WARNING, "PLUGIN: failed to close (%s)",
			     lt_dlerror ());
		}
	} else {
		LOG (LOG_DEBUG, "no need to unload plugin (%s)",
		     pluginid2name[plugin_id]);
	}

	plugin_handle[plugin_id] = NULL;
	plugin_active_list[plugin_id] = NULL;

	return 0;
}

static int _plugin_get_decaps_magic (uint32_t * fourcc_in,
				  uint32_t * fourcc_out, char *plugin_name,
				  u_char * buf)
{
	plugin_decaps_t *plugin_decaps;
	plugin_decaps_magic_t *magic;

	plugin_load_path (PLUGIN_ID_DECAPS, plugin_name, fourcc_in, fourcc_out);
	plugin_decaps = plugin_get_active_ops (PLUGIN_ID_DECAPS);
	magic = plugin_decaps->magic;

	if (magic) {
		LOG (LOG_DEBUG, "%s: file: %s", plugin_name,
		     magic->file_ext);
		LOG (LOG_DEBUG, "%s: len: %d", plugin_name, magic->len);

#ifdef DEBUG
		{
			int i;

			for (i = 0; i < magic->len; i++)
				fprintf (stderr, "0x%02x ",
					 magic->magic[i]);
			fprintf (stderr, "\n\n");
		}
#endif

		if (!strncmp (buf, magic->magic, magic->len)) {
			LOG (LOG_DEBUG,
			     "%s matches -> found the right decapsulation for our file",
			     plugin_name);
			return 0;
		}

	} else
		LOG (LOG_ERROR, "no magic provided by plugin %s",
		     plugin_name);

	plugin_unload (PLUGIN_ID_DECAPS);

	return -1;
}

int plugin_get_decaps_magic (char *filename, buf_t * buf)
{
	plugin_input_t *plugin_input;
	plugin_entry_t *entry;
	buf_entry_t buf_entry;

	buf_entry.mem = malloc (20);
	buf_entry.mem_len = 20;

	plugin_input = plugin_get_active_ops (PLUGIN_ID_INPUT);
	plugin_input->seek (plugin_input, 0);
// FIXME: use a valid buffer
	plugin_input->read (NULL, buf_entry.mem, buf_entry.mem_len);

#ifdef DEBUG
	{
		int i;

		for (i = 0; i < 20; i++)
			fprintf (stderr, "0x%02x ", buf_entry.mem[i]);

		putchar ('\n');
	}
#endif

	entry = plugin_get_registered (PLUGIN_ID_DECAPS);

	while (entry) {
		if (_plugin_get_decaps_magic
		    (entry->fourcc_in, entry->fourcc_out, entry->whoami,
		     buf_entry.mem) == 0) {
			free (buf_entry.mem);
			return 0;
		}

		entry = entry->next;
	}

	free (buf_entry.mem);
	return -1;
}

/**
 * This function is responssible for loading plugins,
 * plugins should contains function :
 *
 * - plugin_init()
 *   Which the plugins will use to register itself back
 *   using the pluginRegister callback.
 *
 * - plugin_exit()
 *
 * FIXME: plugins SHOULD NOT initialize themselfs on init time
 * - if they do - fix it, otherwise we need to call
 * plugin_exit here
 * close the plugin again - we have all important informations,
 * so wait until we reinitialize it at run-time
 **/

static int _plugin_test_load (char *plugin_name)
{
	void (*plugin_ext) (void);
	lt_dlhandle handle;

	LOG (LOG_DEBUG, "checking for plugin \"%s\"", plugin_name);

	if (!(handle = _plugin_load (plugin_name)))
		return -1;

	if (!(plugin_ext = lt_dlsym (handle, "plugin_exit"))) {
		LOG (LOG_WARNING,
		     "no plugin_exit - failed to exit %s (%s)",
		     plugin_name, lt_dlerror ());
		goto err;
	}

	plugin_ext ();

      err:
// what about the handle reference counter?
	if (lt_dlclose (handle) < 0) {
		LOG (LOG_WARNING, "PLUGIN: failed to close %s (%s)",
		     plugin_name, lt_dlerror ());
	}

	return 0;
}

/**
 * scanning for available plugins
 * this function loops through all the plugins found in the specified directory
 * and gathers information provided by the plugins using the registerPlugin
 * callback.
 * ... it also buils up a pluginlist, which contains all available plugins,
 * along with some useful and important information
 *
 * TODO ... cache available plugins, - so we just have to dlopen them to get the
 * fourccs when something on the filesystem has been changed (creation-,
 * modification-time, file-size, MD5?)
 * and produce the plugin-list directly from the cache
 **/

int plugin_mgr_scandir (const char *dirname)
{
	struct dirent *ent;
	int ret = -1;
	DIR *dir = NULL;

	register_time = 1;

	if (!dirname) {
		LOG (LOG_ERROR, "invalid dirname");
		goto clean_up;
	}

	LOG (LOG_DEBUG, "Scanning \"%s\" for plugins", dirname);

	if (!(dir = opendir (dirname))) {
		LOG (LOG_ERROR, "cannot open dir %s (%s)", dirname,
		     strerror (errno));
		goto clean_up;
	}

	while ((ent = readdir (dir))) {
		char *plugin_name;
		char *ext;

		LOG (LOG_DEBUG, "checking plugin: %s", ent->d_name);
		if (!(ext = strrchr (ent->d_name, '.')))
			continue;

		if (strncasecmp (ext, ".so", 3))
			continue;

		LOG (LOG_DEBUG, "test-loading plugin: %s", ent->d_name);
		plugin_name =
		    malloc (strlen (dirname) + strlen (ent->d_name) + 2);
		sprintf (plugin_name, "%s/%s", dirname, ent->d_name);

#if USE_DEPEND
		// clear depend entries
		depend_first = NULL;
		depend_prev = NULL;
#endif

		_plugin_test_load (plugin_name);

#if USE_DEPEND
		// set first depend entry
		if (depend_first != depend_prev)
			depend_first->depend = depend_prev;
#endif

		free (plugin_name);
	}

	ret = 0;
	closedir (dir);

      clean_up:

	register_time = 0;
	return ret;
}

int plugin_init_nav (char *path)
{
	plugin_nav_t *plugin_nav = plugin_get_active_ops (PLUGIN_ID_NAV);

	LOG (LOG_DEBUG, "init nav with path: %s", path);

	return plugin_nav->open (plugin_nav, path);
}

