/*****
*
* This file is part of the OMS program.
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by 
* the Free Software Foundation; either version 2, or (at your option)
* any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; see the file COPYING.  If not, write to
* the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
*
*****/

#define AUTO_DROP_FRAMES 1

#include <pthread.h>

#include <oms/oms.h>
#include <oms/plugin.h>
#include <oms/plugin/decaps.h>
#include <oms/plugin/codec.h>
#include <oms/plugin/output_audio.h>
#include <oms/buf.h>
#include <oms/log.h>

#include "thread_input.h"
#include "thread_demux.h"

static struct {
	pthread_mutex_t mutex;
	pthread_cond_t cond;
	pthread_t thread;

	int status;
} ctrl;


/*
 * Return value :
 * 0 means all is ok,
 * > 0 means we need to drop frame.
 */

static inline int check_output_audio (buf_t *buf)
{
        int ret;
        buf_entry_t *entry;
        plugin_codec_t *plugin_codec;
        plugin_output_audio_t *plugin_output_audio;

        plugin_output_audio = plugin_get_active_ops (PLUGIN_ID_OUTPUT_AUDIO);

        if ((ret = plugin_output_audio->check()) == 2) {
                if ((entry = oms_buf_get (buf, BUF_AUDIO))) {
                        plugin_codec = plugin_get_active_ops (PLUGIN_ID_CODEC_AUDIO);
                        plugin_codec->read (plugin_codec, buf, entry);
                        oms_buf_free (buf, entry);
                }
        }

        return ret;
}


static int _demux (buf_t *buf)
{
        buf_entry_t *buf_entry = NULL;
        plugin_codec_t *plugin_codec;

#ifdef AUTO_DROP_FRAMES
        int drop_flag;
	drop_flag = check_output_audio (buf);
#endif

        buf_entry = oms_buf_get(buf, BUF_ANY);
        if ( ! buf_entry )
                return -1;

        if ( buf_entry->buf_id == BUF_VIDEO ) {
                plugin_codec = plugin_get_active_ops(PLUGIN_ID_CODEC_VIDEO);
#ifdef AUTO_DROP_FRAMES
                plugin_codec->ctrl (plugin_codec, CTRL_VIDEO_DROP_FRAME, drop_flag);
#endif
        }

        else if ( buf_entry->buf_id == BUF_AUDIO ) 
                plugin_codec = plugin_get_active_ops(PLUGIN_ID_CODEC_AUDIO);
                
        else if ( buf_entry->buf_id == BUF_SUBPIC )
                plugin_codec = plugin_get_active_ops(PLUGIN_ID_CODEC_SPU);

        else {
                oms_buf_free(buf, buf_entry);
                return -1;
        }

        
        plugin_codec->read(plugin_codec, buf, buf_entry);
        oms_buf_free(buf, buf_entry);

        return 0;
}

static void *_thread_demux (void *_buf)
{
	buf_t *buf = (buf_t *) _buf;

// prefill buffer
        while (oms_buf_needs_refill (buf))
                usleep (100);

        while (ctrl.status != STATUS_STOP) {
                if (ctrl.status == STATUS_PAUSE) {
                        pthread_mutex_lock(&ctrl.mutex);
                        while ( ctrl.status == STATUS_PAUSE )
                                pthread_cond_wait (&ctrl.cond, &ctrl.mutex);
                        pthread_mutex_unlock(&ctrl.mutex);
                }       

                if (oms_buf_needs_refill (buf))
                        oms_thread_input_kick ();

                if (!oms_buf_isempty (buf))
                        _demux (buf);
        }

        pthread_exit (NULL);
        return NULL;
}

void oms_thread_demux_start (buf_t *buf)
{
        pthread_mutex_init (&ctrl.mutex, NULL);
        pthread_cond_init (&ctrl.cond, NULL);
        pthread_create (&ctrl.thread, NULL, _thread_demux, buf);
}

void oms_thread_demux_stop (void)
{
        pthread_cancel (ctrl.thread);
        pthread_cond_destroy(&ctrl.cond);
        pthread_mutex_destroy(&ctrl.mutex);
        pthread_join (ctrl.thread, NULL);
}

void oms_thread_demux_kick (void)
{
        pthread_mutex_lock(&ctrl.mutex);
        pthread_cond_signal (&ctrl.cond);
        pthread_mutex_unlock(&ctrl.mutex);
}

void oms_thread_demux_set_status (int status)
{
	ctrl.status = status;
}

int oms_thread_demux_get_status (void)
{
	return ctrl.status;
}

int oms_thread_demux_is_status (int status)
{
	return ctrl.status == status;
}
