/*****
*
* This file is part of the OMS program.
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by 
* the Free Software Foundation; either version 2, or (at your option)
* any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; see the file COPYING.  If not, write to
* the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
*
*****/

#if HAVE_CONFIG_H
# include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <inttypes.h>

#include <oms/log.h>
#include <oms/plugin/codec.h>
#include <oms/plugin/output_video.h>

#include "libmpeg2/mpeg2.h"

static int _mpeg2dec_open	(void *self, void *foo);
static int _mpeg2dec_close	(void *self);
static int _mpeg2dec_read	(void *self, buf_t *buf, buf_entry_t *buf_entry);
static int _mpeg2dec_ctrl	(void *self, uint flag, ...);

static mpeg2dec_t _mpeg2dec_priv;

static plugin_codec_t codec_mpeg2dec = {
	priv:		&_mpeg2dec_priv,
        open:		_mpeg2dec_open,
        close:		_mpeg2dec_close,
        read:		_mpeg2dec_read,
	ctrl:		_mpeg2dec_ctrl,
        config:         NULL,
};


static int _mpeg2dec_open (void *self, void *foo)
{
	mpeg2dec_t *priv = ((plugin_codec_t *) self)->priv;
        vo_output_video_t *p = priv->output;

        LOG(LOG_DEBUG, "OPEN, output ptr at %p\n", p);
        
	mpeg2_init (priv, p, NULL);
	return 0;
}


static int _mpeg2dec_close (void *self)
{
	mpeg2dec_t *priv = ((plugin_codec_t *) self)->priv;

	priv->output = ((plugin_codec_t *) self)->output;
	mpeg2_close (priv);
	return 0;
}


static int _mpeg2dec_read (void *self, buf_t *buf, buf_entry_t *buf_entry)
{
	mpeg2dec_t *priv = ((plugin_codec_t *) self)->priv;

	priv->output = ((plugin_codec_t *) self)->output;
	mpeg2_decode_data (priv, buf_entry->data, buf_entry->data+buf_entry->data_len);
        return 0;
}


static int _mpeg2dec_ctrl (void *self, uint ctrl_id, ...)
{
	va_list arg_list;
	mpeg2dec_t *priv = ((plugin_codec_t *) self)->priv;

	va_start (arg_list, ctrl_id);
	switch (ctrl_id) {
		case CTRL_VIDEO_INITIALIZED: {
			int val = va_arg (arg_list, int);
			mpeg2_output_init (priv, val);
			break;
		}
		case CTRL_VIDEO_DROP_FRAME: {
			int val = va_arg (arg_list, int);
//			fprintf (stderr, "%c", val ? '-':'+');
			mpeg2_drop (priv, val);
			break;
		}
		default:
			va_end (arg_list);
			return -1;
	}

	va_end (arg_list);
	return 0;
}


/*****************************************/


int plugin_init (char *whoami)
{
	pluginRegister (whoami,
		PLUGIN_ID_CODEC_VIDEO,
		"mpg1;mpg2",
		NULL,
		NULL,
		&codec_mpeg2dec);

	return 0;
}


void plugin_exit (void)
{
}
