//#define DENT_DEBUG

/*
 * PROGRAM STREAM DECAPSULATION
 *
 * Copyright (C) 1999-2001  Thomas Mirlacher
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * The author may be reached as <dent@linuxvideo.org>
 *
 *------------------------------------------------------------
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <unistd.h>
#include <inttypes.h>
#include <sys/stat.h>
#include <netinet/in.h>

#include <dvb/ps.h>
#include <dvb/pes.h>
#include <oms/oms.h>
#include <oms/plugin/decaps.h>

static int _ps_open (void *self, void *name);
static int _ps_close (void *self);
static int _ps_decaps (buf_t * buf);
static int _ps_ctrl (void *self, uint ctrl_id, ...);

#define BLOCKSIZE_READ	0x800
// alloc 1 page
#define BLOCKSIZE_ALLOC	BLOCKSIZE_READ
//#define BLOCKSIZE_ALLOC       (1<<16)-1

static struct ps_priv_struct {
	uint filter[_BUF_MAX];
} priv;

static uint8_t ps_file_magic[] = {
	0x00, 0x00, 0x01, 0xBA
};

static plugin_decaps_magic_t _ps_magic = {
	"ps;vob",
	4,
	ps_file_magic
};

static plugin_decaps_t decaps = {
	open:_ps_open,
	close:_ps_close,
	decaps:_ps_decaps,
	ctrl:_ps_ctrl,
	magic:&_ps_magic,
	blocksize:BLOCKSIZE_READ,
	config:NULL,
};

static buf_entry_t *buf_entry = NULL;
static buf_t *buf;

#define PS_FILTER_VIDEO	0xE0
#define PS_FILTER_SPU	0x20

static int _ps_open (void *self, void *name)
{
	priv.filter[BUF_VIDEO] = PS_FILTER_VIDEO;
	priv.filter[BUF_AUDIO] = 0;	// audio substream id (0...7, -1=off)
	priv.filter[BUF_SUBPIC] = -1;	// subtitle substream id (0..31,-1=off)

	return 0;
}

static int _ps_close (void *self)
{
	return 0;
}

static inline uint8_t *_get_bytes_block (uint len)
{
	if (decaps.read (NULL, buf_entry->mem, decaps.blocksize) <= 0) {
		return NULL;
	}

	buf_entry->data = buf_entry->mem;
	buf_entry->data_len = buf_entry->mem_len;

	LOG (LOG_DEBUG, "mem_len: %x", buf_entry->mem_len);

	return buf_entry->mem;
}

static inline void _unget_bytes (uint len)
{
	uint8_t *ptr = buf_entry->data;

	buf_entry->data = ptr - len;
}

static inline uint8_t *_get_bytes (uint len)
{
	uint8_t *ptr = buf_entry->data;

	buf_entry->data = ptr + len;

	return ptr;
}

static inline uint8_t *_get_bytesX (uint len)
{
	uint8_t *ptr = buf_entry->data;

	buf_entry->data += len;
	buf_entry->data_len -= len;

	return ptr;
}

static inline uint8_t _get_pes (void)
{
	pes_hdr_t *pes;
	pes_flag_t *flag;

	_unget_bytes (PS_HDR_LEN);
	pes = (pes_hdr_t *) _get_bytes (PES_HDR_LEN);
	flag = (pes_flag_t *) _get_bytes (PES_FLAG_LEN);

	buf_entry->data_len =
	    pes_get_len (pes) - PES_FLAG_LEN -
	    flag->pes_header_data_length;

	_get_bytes (flag->pes_header_data_length);

	if (flag->pts_dts_flag) {
		buf_entry->pts = _pes_get_pts (flag);
		buf_entry->flags |= BUF_FLAG_PTS_VALID;
	}

	return (*buf_entry->data & 0xff);
}

static inline void _write_pes (int bytes)
{
#ifdef DENT_DEBUG
	switch (buf_entry->buf_id) {
	case BUF_SUBPIC:
		LOG (LOG_INFO, "spu");
		break;
	case BUF_AUDIO:
		LOG (LOG_INFO, "audio");
		break;
	case BUF_VIDEO:
		LOG (LOG_INFO, "video");
		break;
	default:
		LOG (LOG_INFO, "unknown");
	}
#endif

	_get_bytesX (bytes);
	decaps.write (buf, buf_entry);
	buf_entry = NULL;
}

static void _handle_pack_hdr (void)
{
	uint8_t *ptr = buf_entry->data;

	if ((*ptr & 0xC0) == 0x40) {	// MPEG 1
		ps_pack_t *ps_pack =

		    (ps_pack_t *) _get_bytes (PS_PACK_LEN);

		buf_entry->scr = _ps_get_scr (ps_pack);
		buf_entry->flags |= BUF_FLAG_SCR_VALID;
		_get_bytes (ps_pack->pack_stuffing_length);
	} else if ((*ptr & 0xF0) == 0x20) {	// MPEG 2
		_get_bytes (8);
	} else			// Unknow header
		LOG (LOG_WARNING, "unknown packet header %x", *ptr);
}

static void _handle_system_hdr (void)
{
	ps_system_hdr_t *ps_system;

	ps_system = (ps_system_hdr_t *) _get_bytes (PS_SYSTEM_HDR_LEN);
	_get_bytes (_ps_get_hdr_len (ps_system) - PS_SYSTEM_HDR_LEN + 2);
}

static uint8_t *_handle_padding (void)
{
	// get new data.
	return _get_bytes_block (PS_HDR_LEN);
}

static int _handle_audio_spu (void)
{
	uint8_t substream_id = _get_pes ();

	buf_entry->buf_id = BUF_AUDIO;

	// This is the spu stream we want.
	if (substream_id == priv.filter[BUF_SUBPIC]) {
		buf_entry->buf_id = BUF_SUBPIC;
		_write_pes (1);
	} else if (substream_id >= PS_FILTER_SPU &&
		   substream_id <= (PS_FILTER_SPU + 31)) {
		return -1;
	} else if ((substream_id & 0x0f) != priv.filter[BUF_AUDIO]) {
		return -1;
	} else if ((substream_id & 0xf0) == PES_SUBSTREAM_AUDIO_AC3) {
		buf_entry->type = FOURCC_AC3;
		_write_pes (4);
	} else if ((substream_id & 0xf0) == PES_SUBSTREAM_AUDIO_LPCM) {
		buf_entry->type = FOURCC_PCM;
		_write_pes (7);
	} else {
		LOG (LOG_ERROR, "unhandled audio stream: %x %x %x %x\n",
		     *(buf_entry->data + 0), *(buf_entry->data + 1),
		     *(buf_entry->data + 2), *(buf_entry->data + 3));
		return -1;
	}

	return 0;
}

static void _handle_pci_dsi (void)
{
	_get_pes ();
	buf_entry->type = FOURCC_NAV;
	buf_entry->buf_id = BUF_OTHER;
	_write_pes (0);
}

static int _handle_unknown (ps_hdr_t * ps)
{
	_get_pes ();

	if (ps->stream_id == priv.filter[BUF_VIDEO]) {
		buf_entry->buf_id = BUF_VIDEO;
		buf_entry->type = FOURCC_MPEG;
		_write_pes (0);
	} else if (ps->stream_id == (0xC0 + priv.filter[BUF_AUDIO])) {
		buf_entry->buf_id = BUF_AUDIO;
		buf_entry->type = FOURCC_MPEG1;
		_write_pes (0);
	} else {
		LOG (LOG_ERROR, "unhandled stream_id %x\n", ps->stream_id);
		return -1;
	}

	return 0;
}

static inline int _find_startcode (ps_hdr_t * ps)
{
	uint8_t *ptr;

	do {			// find byte-aligned startcode
		if (!(ptr = _get_bytes (PS_HDR_LEN)))
			return -1;

		memcpy (ps, ptr, PS_HDR_LEN);

		if (_pes_is_startcode (ps->start_code_prefix))
			break;

		//LOG (LOG_ERROR, "invalid startcode (%06x)", ps->start_code_prefix);
		_unget_bytes (PS_HDR_LEN - 1);
	} while (1);

	return 0;
}

static int _ps_decaps (buf_t *_buf)
{
	int ret = 0;
	ps_hdr_t ps;
	buf = _buf;

	if (!buf_entry)
		if ((buf_entry = oms_buf_alloc (buf, BLOCKSIZE_ALLOC)))
			return 0;

	// get new block.
	if (!(_get_bytes_block (PS_HDR_LEN)))
		return -1;

	buf_entry->flags = BUF_FLAG_NONE;

	while (buf_entry) {
		if (_find_startcode (&ps))
			goto err;

		switch (ps.stream_id) {
		case PS_STREAM_PACK_HDR:
			_handle_pack_hdr ();
			break;

		case PS_STREAM_SYSTEM_HDR:
			_handle_system_hdr ();
			break;

		case PS_STREAM_PADDING:
			if (!_handle_padding ())
				goto err;
			break;

		case PES_STREAM_AUDIO_SPU:
			if ((ret = _handle_audio_spu ()) < 0)
				goto err;
			break;

		case PES_STREAM_PCI_DSI:
			_handle_pci_dsi ();
			break;

		default:
			if ((ret = _handle_unknown (&ps)) < 0)
				goto err;
		}
	}

err:
	return 0;
}

static int _ps_ctrl (void *self, uint ctrl_id, ...)
{
	int filter_id;
	uint32_t val;

	va_list arg_list;

	LOG (LOG_DEBUG, "ctrl_id: 0x%x", ctrl_id);

	va_start (arg_list, ctrl_id);

	switch (ctrl_id) {

	case CTRL_DECAPS_SETFILTER:
		filter_id = va_arg (arg_list, int);

		val = va_arg (arg_list, uint32_t);

		if (filter_id > _BUF_MAX)
			return -1;

		if (val < 0) {
			val = 0xffff;
		} else {
			switch (filter_id) {

			case BUF_VIDEO:
				val += PS_FILTER_VIDEO;
				break;

			case BUF_AUDIO:
				break;

			case BUF_SUBPIC:
				val += PS_FILTER_SPU;
				break;
			}
		}

		priv.filter[filter_id] = val;
		break;

	default:
		va_end (arg_list);
		return -1;
	}

	va_end (arg_list);

	return 0;
}

int PLUGIN_INIT (decaps_ps) (char *whoami) {
	pluginRegister (whoami,
			PLUGIN_ID_DECAPS, "ps", NULL, NULL, &decaps);

	return 0;
}

void PLUGIN_EXIT (decaps_ps) (void) {
}
