#define USE_DVDDB
//#define DENT_DVDNAVDUMP

/*
 *
 * Copyright (C) 1999-2000  Thomas Mirlacher
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * The author may be reached as <dent@linuxvideo.org>
 *
 *------------------------------------------------------------
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <dlfcn.h>
#include <inttypes.h>
#include <sys/stat.h>
#include <netinet/in.h>

#include <bswap.h>
#include <oms/oms.h>		// for callback
#include <oms/log.h>
#include <oms/plugin/input.h>
#include <oms/plugin/decaps.h>
#include <oms/plugin/nav.h>
#include <oms/plugin/codec.h>

#include "udf/dvd_udf.h"
#include "ifo/libifo/ifo.h"
#include "nav_dvd_pkt.h"

//int fd_debug;

static int _dvd_open (void *plugin, void *path);
static int _dvd_close (void *plugin);
static int _dvd_play (int title, int chapter, int parts2play);
static int _dvd_RegisterCb (uint id, oms_callback_t cb, void *data);
static int _dvd_UnregisterCb (uint id);
static nav_tree_t *_dvd_get_info (uint id);

static int _dvd_ifo_init (int fd);
static int _dvd_setup_audio (plugin_nav_t * self, uint8_t stream_id);
static int _dvd_setup_video (plugin_nav_t * self);
static int _dvd_setup_spu (plugin_nav_t * self, uint8_t stream_id);

#ifndef MAX
#define MAX(a,b) ((a)>(b)?(a):(b))
#endif
#ifndef MIN
#define MIN(a,b) ((a)<(b)?(a):(b))
#endif

#ifdef USE_DVDDB
#include <dvddb/dvddb.h>
#endif

#define FILE_IFO "/VIDEO_TS/VIDEO_TS.IFO"

typedef struct dvd_priv_struct {
	const pgc_t *pgci;	// program chain information

	const ifo_pgci_caddr_t *cpli;	// cell play information
	ifo_pgc_cpos_t *cpos;	// cell position pointer
	int32_t cpos_index;	// cell position_index
	const ifo_caddr_t *caddr;	// cell address pointer
	int32_t caddr_index;	// cell address index
	uint32_t caddr_index_max;	// cell address index maximum
	uint32_t cellp_pos;	// cellpiece position
	uint32_t cell_end;	// cell end

	uint32_t *vobuaddr;	// video object address map
	uint32_t vobuaddr_index;	// video object unit address index
	uint32_t vobuaddr_index_max;	// video object unit address index maximum
	uint32_t vobu_end;	// next video unit object address

	off_t title_start;	// start of title (file)
	ifo_t *ifo;		// information structure
	char *path;		// path to device (dir in the future)

	int async_change;	// flag to check if we're allowed to interrupt NOW
	struct {
		void (*chapter) (void *data, int chapter);
		void *chapter_data;
	} callb;		// Callbacks

	struct {
		uint32_t menu_language_code;
		uint32_t audio_stream_number;
		uint32_t subpicture_stream_number;
		uint32_t angle_number;
		uint32_t vts_number;
		uint32_t title_number;
		uint32_t pgc_number;
		uint32_t part_of_title_number;
		uint32_t highlighted_button_number;
		uint32_t navigation_timer;
		uint32_t pgc_jump_for_navigation_timer;
		uint32_t karaoke_audio_mixing_mode;
		uint32_t parental_management_country_code;
		uint32_t parental_level;
		uint32_t video_setting;
		uint32_t audio_setting;
		uint32_t audio_language_code_setting;
		uint32_t audio_language_extension_code;
		uint32_t subpicture_language_code_setting;
		uint32_t subpicture_language_extension_code;
	} reg;
} dvd_priv_t;

static dvd_priv_t dvd_priv;

static int _dvd_next_cell (dvd_priv_t * priv, uint32_t cpos_index);
static int _dvd_pre_read (void *self, void *buf, size_t count);

#ifdef DENT_DVDNAVDUMP
static int _dvd_post_write (buf_t * buf, buf_entry_t * buf_entry);
#endif
static int _dvd_setup_first_block (dvd_priv_t * priv);

static plugin_nav_t dvd_nav = {
	priv:&dvd_priv,
	open:_dvd_open,
	close:_dvd_close,
	play:_dvd_play,
	get_info:_dvd_get_info,
	register_cb:_dvd_RegisterCb,	// register callback
	unregister_cb:_dvd_UnregisterCb,	// unregister callback
	config:NULL,
};

static int _dvd_open (void *self, void *path)
{
	struct dvd_priv_struct *priv = &dvd_priv;
	plugin_input_t *plugin_input;
	plugin_decaps_t *plugin_decaps;
	int fd;

	priv->path = path;
	priv->async_change = 0;
	priv->cellp_pos = 0;
	priv->cell_end = 0;
	priv->vobu_end = 0;

//fd_debug = open ("/tmp/navdvd", O_WRONLY);

	if (!(plugin_input = pluginLoad (PLUGIN_ID_INPUT, "dvd ", NULL))) {
		LOG (LOG_ERROR, "failed to open input plugin");
		return -1;
	}
	plugin_input->blocksize = 2048;
	plugin_input->open (plugin_input, path);

	if (!(plugin_decaps = pluginLoad (PLUGIN_ID_DECAPS, "ps  ", NULL))) {
		LOG (LOG_ERROR, "failed to open decaps plugin");
		return -1;
	}
	plugin_decaps->blocksize = 2048;
	plugin_decaps->read = _dvd_pre_read;
#ifdef DENT_DVDNAVDUMP
	plugin_decaps->write = _dvd_post_write;
#else
	plugin_decaps->write = oms_buf_validate;
#endif
	plugin_decaps->open (plugin_decaps, NULL);

	/*
	 * DENT: check if we've a disc or a directory
	 */
	if ((fd = UDFOpenDisc (path)) < 0) {
		LOG (LOG_ERROR, "unable to open udf-disc %s",
		     (char *) path);
		return -1;
	}

	if (_dvd_ifo_init (fd) < 0) {
		LOG (LOG_ERROR, "failed to initialize IFO");
		return -1;
	}

	close (fd);

	/*
	 * we need to close and reopen, cause we request access for a certain area
	 * of the disk each time we open - do a seek first to position to filepointer
	 * to the right offset ...
	 */
	plugin_input->close (plugin_input);
	plugin_input->seek (plugin_input, priv->title_start);
	plugin_input->open (plugin_input, path);

	_dvd_setup_audio ((plugin_nav_t *) self, 0);
	_dvd_setup_video ((plugin_nav_t *) self);
	_dvd_setup_spu ((plugin_nav_t *) self, 0);

	return 0;
}

static int _dvd_close (void *plugin)
{
	struct dvd_priv_struct *priv = &dvd_priv;

	ifoClose (priv->ifo);
	priv->ifo = NULL;

	return 0;
}

static int _dvd_play (int title_number, int part_of_title_number,
		      int parts2play)
{
	struct dvd_priv_struct *priv = &dvd_priv;

	priv->reg.title_number = title_number;
	priv->reg.part_of_title_number = part_of_title_number;

	priv->async_change = 1;
	return 0;
}

static int _dvd_RegisterCb (uint id, oms_callback_t cb, void *data)
{
	struct dvd_priv_struct *priv = &dvd_priv;

	LOG (LOG_DEBUG, "UI registered callback");

	switch (id) {
	case CB_BUF_DONE:
		priv->callb.chapter = cb;
		priv->callb.chapter_data = data;
		break;
	default:
		return -1;
	}

	return 0;
}

static int _dvd_UnregisterCb (uint id)
{
	LOG (LOG_DEBUG, "UI unregistered callback %d", id);

	return 0;
}

/*
 * "Title " + ?? + '\0'
 */
#define TITLE_NAME_LEN (6 + 2 + 1)

/*
 * 'Ghostbuster fix' by Yuqing Deng <deng@tinker.chem.brown.edu>
 * DENT: i absolutely don't know why we need 8 instead of 7 below ...
 * "Chapter " + ??? + '\0'
 */
#define PART_NAME_LEN (8 + 3 + 1)

#ifdef USE_DVDDB
static FILE *_dvddb_get (char dvd_id[9])
{
	int sock,
	 answer;
	FILE *dvddb_file;

	dvddbGetID (dvd_priv.path, dvd_id);

	dvddb_file = dvddbOpen (dvd_id);
	if (dvddb_file)
		return dvddb_file;

	LOG (LOG_DEBUG, "not found in local cache.");
	fprintf (stderr,
		 "should i try to fetch it from the internet? (Y/n) : ");

	answer = getchar ();
	if (answer < 0)
		perror ("getchar()");

	if (answer == 'n' || answer == 'N')
		return NULL;

	sock = dvddbNetConnect ();
	if (sock < 0)
		return NULL;

	dvddbNetGet (sock, dvd_id);
	dvddbNetDisconnect (sock);

	if (!(dvddb_file = dvddbOpen (dvd_id))) {
		LOG (LOG_INFO,
		     "--------------------------------------------------");
		LOG (LOG_INFO,
		     "did not find an entry in the online-database,");
		LOG (LOG_INFO,
		     "but it would really great f you submit this entry.");
		LOG (LOG_INFO,
		     "(see src/FAQ: how do i submit new entries to the DVDDB?)");
		LOG (LOG_INFO, "- thanks in advance for your help");
		LOG (LOG_INFO,
		     "--------------------------------------------------");
	}

	return dvddb_file;
}
#endif

static nav_tree_t *_get_info_audio (void)
{
	nav_tree_t *tree;
	ifo_audio_t *audio;
	int num;
	int i;

	num = ifoGetAudio (dvd_priv.ifo->tbl[ID_MAT] + IFO_OFFSET_AUDIO,
			   (uint8_t **) & audio);

	if (num <= 0)
		return NULL;

	if (num > 0)
		num++;

	tree = malloc (sizeof (nav_tree_t));
	tree->num = num;
	tree->title = malloc (sizeof (nav_subtree_t) * tree->num);

	for (i = 0; i < num; i++) {
		nav_subtree_t *sub_tree = tree->title + i;

		sub_tree->sub = NULL;
		sub_tree->num = 0;

		if (!i)
			sub_tree->name = strdup ("(off)");
		else {
			sub_tree->name =
			    strdup (ifoDecodeLang (audio->lang_code));
			audio++;
		}
	}

	return tree;
}

static nav_tree_t *_get_info_subpic (void)
{
	nav_tree_t *tree;
	ifo_spu_t *spu;
	int num;
	int i;

	num = ifoGetSPU (dvd_priv.ifo->tbl[ID_MAT] + IFO_OFFSET_SUBPIC,
			 (uint8_t **) & spu);

	if (num <= 0)
		return NULL;

	if (num > 0)
		num++;

	tree = malloc (sizeof (nav_tree_t));
	tree->num = num;
	tree->title = malloc (sizeof (nav_subtree_t) * tree->num);

	for (i = 0; i < num; i++) {
		nav_subtree_t *sub_tree = tree->title + i;

		sub_tree->sub = NULL;
		sub_tree->num = 0;

		if (!i)
			sub_tree->name = strdup ("(off)");
		else {
			sub_tree->name =
			    strdup (ifoDecodeLang (spu->lang_code));
			spu++;
		}
	}

	return tree;
}

static nav_tree_t *_get_info_program (void)
{
	int i,
	 s;
	nav_tree_t *tree;
	ifo_ptt_t *ptt;

#ifdef USE_DVDDB
	FILE *dvddb_file;
	char dvd_id[9];
#endif

	if (!(ptt = ifoGetPTT (dvd_priv.ifo))) {
		LOG (LOG_ERROR, "failed to get PTT (Part of Title)");
		return NULL;
	}

	tree = malloc (sizeof (nav_tree_t));
	tree->num = ptt->num;
	tree->title = malloc (sizeof (nav_subtree_t) * tree->num);

#ifdef USE_DVDDB
	dvddb_file = _dvddb_get (dvd_id);
#endif

	LOG (LOG_DEBUG, "get_info: ptt_num: %d", ptt->num);

	for (i = 0; i < ptt->num; i++) {
		ifo_ptt_sub_t *title = ptt->title + i;
		nav_subtree_t *sub_tree = tree->title + i;
		char *name;

		sub_tree->sub =
		    (nav_tree_data_t *) malloc (sizeof (nav_tree_data_t) *
						title->num);
		sub_tree->num = title->num;

#ifdef USE_DVDDB
		if ((name = dvddbRead (dvddb_file)))
			sub_tree->name = strdup (name);
		else
#endif
		{
			name = (char *) malloc (TITLE_NAME_LEN);
			snprintf (name, TITLE_NAME_LEN, "Title %d", i + 1);
			sub_tree->name = name;
		}

#ifdef USE_DVDDB
		dvddbRead (dvddb_file);	// read blank line after title
#endif
		for (s = 0; s < title->num; s++) {
			nav_tree_data_t *data = sub_tree->sub + s;
			char *name;

#ifdef USE_DVDDB
			if ((name = dvddbRead (dvddb_file)))
				data->name = strdup (name);
			else
#endif
			{
				name = (char *) malloc (PART_NAME_LEN);
				snprintf (name, PART_NAME_LEN,
					  "Chapter %d", s + 1);
				data->name = name;
			}
		}
#ifdef USE_DVDDB
		/*
		 * FIXME: check if that breaks something ...
		 */
		dvddbClose (dvddb_file);
#endif
	}

	return tree;
}

static nav_tree_t *_dvd_get_info (uint id)
{
	nav_tree_t *tree = NULL;

	switch (id) {
	case OMS_INFO_PROGRAM:{
			tree = _get_info_program ();
			break;
		}
	case OMS_INFO_AUDIO:{
			tree = _get_info_audio ();
			break;
		}

	case OMS_INFO_SUBPIC:{
			tree = _get_info_subpic ();
			break;
		}
	default:
		LOG (LOG_DEBUG, "unhandled OMS_INFO %d", id);
	}

	return tree;
}

static int _dvd_ifo_init (int fd)
{
	struct dvd_priv_struct *priv = &dvd_priv;

	if (priv->ifo) {
		ifoClose (priv->ifo);
		priv->ifo = NULL;
	}
// find VIDEO_TS.IFO
	if (!(priv->title_start = UDFFindFile (FILE_IFO))) {
		LOG (LOG_DEBUG, "unable to find UDFFile %s", FILE_IFO);
		return -1;
	}
// opening VIDEO_TS.IFO
	if (!
	    (priv->ifo =
	     ifoOpen (fd, priv->title_start * DVD_VIDEO_LB_LEN))) {
		LOG (LOG_ERROR, "error initializing ifo");
		return -1;
	}
// FIXME: temporary hack until we've the menu system up and running
//      btw. this hack is wrong
// FIXME: get autoplay here ...
// getting offset of first important ifo file (from TitleSetPointers)
	priv->title_start += ifoGetTSPoffset (priv->ifo, 0);

// closing VIDEO_TS.IFO
	ifoClose (priv->ifo);
	priv->ifo = NULL;

// opening VTS_??_0.IFO
	if (!
	    (priv->ifo =
	     ifoOpen (fd, priv->title_start * (off_t) DVD_VIDEO_LB_LEN))) {
		LOG (LOG_ERROR, "error initializing ifo");
		return -1;
	}

	priv->caddr = ifoGetCellAddr (priv->ifo);
	priv->title_start += (off_t) ifoGetVOBStart (priv->ifo);

	return 0;
}

static int _dvd_setup_audio (plugin_nav_t * self, uint8_t stream_id)
{
	plugin_codec_t *plugin_audio;
	struct dvd_priv_struct *priv = &dvd_priv;
	ifo_audio_t *audio;
	plugin_codec_attr_audio_t attr;

// DENT: this sets the audio for the first stream - change again on stream change !!!  
	ifoGetAudio (priv->ifo->tbl[ID_MAT] + IFO_OFFSET_AUDIO,
		     (uint8_t **) & audio);

	audio += stream_id;

	if (!audio->coding_mode && !audio->num_channels)
		attr.mode = 2;	// LPCM
	else
		attr.mode = audio->coding_mode;

	attr.bits = audio->quantization;
	attr.freq = audio->sample_freq ? 2 : 1;	// 96kHz : 48kHz
	attr.stream = stream_id;
	attr.volume = -1;

	LOG (LOG_DEBUG, "audio_coding_mode: %x\n", audio->coding_mode);

	switch (audio->coding_mode) {
	case 0x00:		// AC3
		if (!pluginLoad (PLUGIN_ID_CODEC_AUDIO, "ac3 ", NULL)) {
			LOG (LOG_ERROR,
			     "failed to open ac3 audio codec plugin");
			return -1;
		}
		break;
	case 0x02:		// MPEG1
	case 0x03:		// MPEG2 ext
		if (!pluginLoad (PLUGIN_ID_CODEC_AUDIO, "mpg1", NULL)) {
			LOG (LOG_ERROR,
			     "failed to open mpg123 audio codec plugin");
			return -1;
		}
		break;
	case 0x04:		// LPCM
		if (!pluginLoad (PLUGIN_ID_CODEC_AUDIO, "pcm ", NULL)) {
			LOG (LOG_ERROR,
			     "failed to open pcm audio codec plugin");
			return -1;
		}
		break;
	default:
		LOG (LOG_ERROR, "unhandled audio type %d",
		     audio->coding_mode);
		return -1;
	}

	if ((plugin_audio = plugin_get_active_ops (PLUGIN_ID_CODEC_AUDIO))) {
		plugin_audio->output = plugin_get_active_ops (PLUGIN_ID_OUTPUT_AUDIO);
		plugin_audio->open (plugin_audio, NULL);
//              plugin_audio->set_attributes (BUF_AUDIO, &attr);
	}

	return 0;
}

static int _dvd_setup_video (plugin_nav_t * self)
{
	plugin_codec_t *plugin_video;

	struct dvd_priv_struct *priv = &dvd_priv;
	ifo_video_info_t *video;
	plugin_codec_attr_video_t attr;

	video =
	    (ifo_video_info_t *) priv->ifo->tbl[ID_MAT] + IFO_OFFSET_VIDEO;

	// compression
	attr.mode = 0;		// VOB
	attr.fps = video->system;
	attr.aspect_mode = video->perm_displ;
	attr.aspect_output = video->ratio;
	attr.tv_system = -1;

	if (!
	    (plugin_video =
	     pluginLoad (PLUGIN_ID_CODEC_VIDEO, "mpg2", NULL))) {
		LOG (LOG_ERROR, "failed to open video codec plugin");
		return -1;
	}

	plugin_video->output = plugin_get_active_ops (PLUGIN_ID_OUTPUT_VIDEO);
	plugin_video->open (plugin_video, NULL);

//      if (plugin_video->set_attributes)
//              plugin_video->set_attributes (BUF_VIDEO, &attr);

	return 0;
}

static int _dvd_setup_spu (plugin_nav_t * self, uint8_t stream_id)
{
	plugin_codec_t *plugin_spu;

	if (!(plugin_spu = pluginLoad (PLUGIN_ID_CODEC_SPU, "spu ", NULL))) {
		LOG (LOG_ERROR, "failed to open spu codec plugin");
		return -1;
	}

	plugin_spu->output = plugin_get_active_ops (PLUGIN_ID_OUTPUT_VIDEO);
	plugin_spu->open (plugin_spu, NULL);

	return 0;
}

/**
 * setup all needed IFO maps
 **/

static int _dvd_setup_first_block (dvd_priv_t * priv)
{
	const uint8_t title_or_menu = IFO_TITLE;
	const uint8_t *clink;

	if (!
	    (priv->pgci =
	     ifoGetPGCI (priv->ifo, title_or_menu,
			 priv->reg.title_number))) {
		LOG (LOG_ERROR, "error ifoGetPGCI");
		return -1;
	}
// using Cell Link to find first cell of selected chapter
	clink = ifoGetProgramMap (priv->pgci);
	priv->cpos_index = clink[priv->reg.part_of_title_number] - 1;

	priv->caddr_index = -1;
	priv->caddr_index_max =
	    ifoGetCellAddrNum (priv->ifo, title_or_menu);
	priv->cellp_pos = 0;
	priv->cell_end = 0;

// Cell Position
	if (!(priv->cpos = ifoGetCellPos (priv->pgci))) {
		LOG (LOG_ERROR, "error ifoGetCellPos");
		return -1;
	}
// Cell Play Information
	if (!(priv->cpli = ifoGetCellPlayInfo (priv->pgci))) {
		LOG (LOG_ERROR, "error ifoGetCellPlayInfo");
		return -1;
	}
// Video Object Unit Address Map
// DENT: make a nice ifo wrapper here
	priv->vobuaddr = (uint32_t *) priv->ifo->tbl[ID_TITLE_VOBU_ADMAP];
	priv->vobuaddr_index = 0;
	priv->vobuaddr_index_max = *priv->vobuaddr;
	priv->vobuaddr++;	// header is 4 bytes
	priv->vobu_end = 0;

	{
		plugin_codec_t *codec;
		const clut_t *clut;

		codec = plugin_get_active_ops (PLUGIN_ID_CODEC_SPU);

		if (!(clut = ifoGetCLUT (priv->pgci))) {
			LOG (LOG_ERROR, "getting CLUT");
		}

		codec->ctrl (codec, CTRL_SPU_SET_CLUT, clut);
	}

	return 0;
}

/**
 * calculate next cell, using some IFO magic ;)
 **/

static int _dvd_next_cell (dvd_priv_t * priv, const uint32_t cpos_index)
{
	const int32_t caddr_index = priv->caddr_index;

	priv->cpos_index = cpos_index;

	LOG (LOG_DEBUG, "looking for cell %x:%x (%x/%x)\n",
	     priv->cpos[priv->cpos_index].vob_id,
	     priv->cpos[priv->cpos_index].cell_id, cpos_index,
	     priv->caddr_index_max);

	if (cpos_index > priv->caddr_index_max) {
		LOG (LOG_ERROR, "index >max");
		return -1;
	}

	LOG (LOG_DEBUG, "max_index: %x ", priv->caddr_index_max);

	while (++priv->caddr_index < priv->caddr_index_max) {
		LOG (LOG_DEBUG, "cur_index: %x ", priv->caddr_index);
		LOG (LOG_DEBUG, "cell %x:%x (%x:%x)\n",
		     priv->cpos[priv->cpos_index].vob_id,
		     priv->caddr[priv->caddr_index].vob_id,
		     priv->cpos[priv->cpos_index].cell_id,
		     priv->caddr[priv->caddr_index].cell_id);

		if (
		    (priv->cpos[priv->cpos_index].vob_id ==
		     priv->caddr[priv->caddr_index].vob_id)
		    && (priv->cpos[priv->cpos_index].cell_id ==
			priv->caddr[priv->caddr_index].cell_id)) {
			LOG (LOG_DEBUG, "**********FOUND***********\n");
			goto cell_found;
		}
	}

//      LOG (LOG_ERROR, "cell %x:%x not found", priv->cpos[priv->cpos_index].vob_id, priv->cpos[priv->cpos_index].cell_id);
	priv->caddr_index = caddr_index;
	return -1;

      cell_found:
// for ILVUs which don't end at cell boundary
	priv->cellp_pos =
	    MAX (be2me_32 (priv->caddr[priv->caddr_index].start),
		 be2me_32 (priv->cpli[cpos_index].vobu_start));
	priv->cell_end =
	    MIN (be2me_32 (priv->caddr[priv->caddr_index].end),
		 be2me_32 (priv->cpli[cpos_index].vobu_last_end));

	LOG (LOG_DEBUG, "cellp_pos:%x cell_end:%x", priv->cellp_pos,
	     priv->cell_end);
	return 0;
}

/**
 * get next cellpiece (block) from input
 **/

static int _dvd_pre_read (void *self, void *buf, size_t count)
{
	int ret;
	struct dvd_priv_struct *priv = &dvd_priv;
	plugin_input_t *plugin_input = plugin_get_active_ops (PLUGIN_ID_INPUT);

// async (user selected) jump
	if (priv->async_change) {
		_dvd_setup_first_block (priv);
		priv->async_change = 0;
		//oms_buf_flush (buf, BUF_ANY);
	}
// do the day2day business
	if (++priv->cellp_pos > priv->cell_end) {	// next cell
		if (_dvd_next_cell (priv, priv->cpos_index) < 0)
			if (_dvd_next_cell (priv, priv->cpos_index + 1) <
			    0) return 0;

		if (plugin_input->seek (plugin_input,
					priv->title_start +
					priv->cellp_pos) < 0) {
			LOG (LOG_ERROR, "error in seeking\n");
		}
	}
// keep track of video object units
	if (priv->cellp_pos > priv->vobu_end) {
		for (priv->vobuaddr_index++;
		     priv->vobuaddr_index <= priv->vobuaddr_index_max;
		     priv->vobuaddr_index++) {
			if (be2me_32 (priv->vobuaddr[priv->vobuaddr_index])
			    > priv->cellp_pos) {
				priv->vobu_end =
				    be2me_32 (priv->vobuaddr
					      [priv->vobuaddr_index]) - 1;
				break;
			}
		}
	}

	ret = plugin_input->read (self, buf, count);

	if (ret != 0x800) {	// check if we got a whole block
		LOG (LOG_ERROR, "cell_pos:%x unable to get full block (%d)",
		     priv->cellp_pos, ret);
		return -1;
	}

	return ret;
}

#ifdef DENT_DVDNAVDUMP
/**
 * ok, decapsulator called, and returned something, but we do some pre-checking
 * on the data (probably we need something like DSI stuff)
 **/

static int _dvd_post_write (buf_t * buf, buf_entry_t * buf_entry)
{
//if (fd_debug)
//write (fd_debug, buf_entry->mem, buf_entry->mem_len);

	if (buf_entry->type == FOURCC_NAV) {
		decode_nav (buf_entry->mem, buf_entry->mem_len);
		return oms_buf_free (buf, buf_entry);
	}

	return oms_buf_validate (buf, buf_entry);
}
#endif

int plugin_init (char *whoami)
{
	pluginRegister (whoami,
			PLUGIN_ID_NAV, "dvd", NULL, NULL, &dvd_nav);

	return 0;
}

void plugin_exit (void)
{
}
