//PLUGIN_INFO(INFO_NAME, "audio driver output");
//PLUGIN_INFO(INFO_AUTHOR, "Aaron Holtzman <aholtzma@ess.engr.uvic.ca>");

/*
 *
 *  audio_out_sys.c
 *    
 *	Copyright (C) Aaron Holtzman - May 1999
 *	Port to IRIX by Jim Miller, SGI - Nov 1999
 *	Port to output plugin/unified support for linux/solaris/...
 *		by Thomas Mirlacher - Jul 2000
 *
 *  This file is part of ac3dec, a free Dolby AC-3 stream decoder.
 *	
 *  ac3dec is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  ac3dec is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 *
 */

#include <stdio.h>
#include <errno.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <math.h>
#include <sys/ioctl.h>

#if defined (__OpenBSD__)
#	include <soundcard.h>
#elif defined (__FreeBSD__)
#	include <machine/soundcard.h>
#elif defined (__sun__)
#	include <sys/audioio.h>
#	include <stropts.h>
#	include <signal.h>
#elif defined (__irix__)
#	include <audio.h>
#else
#	include <sys/soundcard.h>
#endif

#include <oms/log.h>
#include <oms/plugin/output_audio.h>

// FIXME: move all global variables into a priv struct
static int fd;
static uint32_t bufFill = 0;

#if defined (__irix__)
static ALport alport = 0;
static ALconfig alconfig = 0;
static int bytesPerWord = 1;
static int nChannels = 2;
#endif

static int _audio_sys_open (void *plugin, void *name);
static int _audio_sys_close (void *plugin);
static int _audio_sys_setup (plugin_output_audio_attr_t * attr);
static int _audio_sys_write (const void *buf, size_t num_bytes);
static int _audio_sys_check (void);

static plugin_output_audio_t audio_sys = {
	open:_audio_sys_open,
	close:_audio_sys_close,
	setup:_audio_sys_setup,
	write:_audio_sys_write,
	check:_audio_sys_check,
	config:NULL,
};

static int _audio_sys_open (void *plugin, void *name)
{
#if defined (__irix__)
	ALpv params[2];
	int dev = AL_DEFAULT_OUTPUT;
	int wsize = AL_SAMPLE_16;

	nChannels = attr->channels;

	alconfig = alNewConfig ();

	if (alSetQueueSize (alconfig, BUFFER_SIZE) < 0) {
		fprintf (stderr, "alSetQueueSize failed: %s\n",
			 alGetErrorString (oserror ()));
		return -1;
	}

	if (alSetChannels (alconfig, attr->channels) < 0) {
		fprintf (stderr, "alSetChannels(%d) failed: %s\n",
			 attr->channels, alGetErrorString (oserror ()));
		return -1;
	}

	if (alSetDevice (alconfig, dev) < 0) {
		fprintf (stderr, "alSetDevice failed: %s\n",
			 alGetErrorString (oserror ()));
		return -1;
	}

	if (alSetSampFmt (alconfig, AL_SAMPFMT_TWOSCOMP) < 0) {
		fprintf (stderr, "alSetSampFmt failed: %s\n",
			 alGetErrorString (oserror ()));
		return -1;
	}

	alport = alOpenPort ("AC3Decode", "w", 0);

	if (!alport) {
		fprintf (stderr, "alOpenPort failed: %s\n",
			 alGetErrorString (oserror ()));
		return -1;
	}

	switch (attr->bits) {
	case 8:
		bytesPerWord = 1;
		wsize = AL_SAMPLE_8;
		break;

	case 16:
		bytesPerWord = 2;
		wsize = AL_SAMPLE_16;
		break;

	case 24:
		bytesPerWord = 4;
		wsize = AL_SAMPLE_24;
		break;

	default:
		fprintf (stderr, "Irix audio: unsupported bit with %d\n",
			 attr->bits);
		break;
	}

	if (alSetWidth (alconfig, wsize) < 0) {
		fprintf (stderr, "alSetWidth failed: %s\n",
			 alGetErrorString (oserror ()));
		return -1;
	}

	params[0].param = AL_RATE;
	params[0].value.ll = alDoubleToFixed ((double) attr->rate);
	params[1].param = AL_MASTER_CLOCK;
	params[1].value.i = AL_CRYSTAL_MCLK_TYPE;
	if (alSetParams (dev, params, 1) < 0) {
		printf ("alSetParams() failed: %s\n",
			alGetErrorString (oserror ()));
		return -1;
	}
#elif defined (__sun__)
	// Open the device driver

	if ((fd = open ((char *) name, O_WRONLY)) < 0) {
		fprintf (stderr, "%s: Opening audio device %s\n",
			 strerror (errno), (char *) name);
		return -1;
	}
	fprintf (stderr, "Opened audio device \"%s\"\n", (char *) name);

#else
// Open the device driver
	struct audio_buf_info foo;
	int tmp;

	//first we try to open it with O_NONBLOCK, if that succeeds, close it then open it normally
	if ((fd = open ((char *) name, O_WRONLY|O_NONBLOCK)) < 0) {
		fprintf (stderr, "%s: Opening audio device %s\n",
			 strerror (errno), (char *) name);
		LOG(LOG_WARNING, "Your audio device failed to open, check to see that it is set up correctly and that another program is not accessing it(ie, xmms or esd).");
		return -1;
	}
	//close the dsp so we can open it normally
	close(fd);
     	if ((fd = open ((char *) name, O_WRONLY)) < 0) {
		fprintf (stderr, "%s: Opening audio device %s\n",
			 strerror (errno), (char *) name);
		return -1;
	}
	
	fprintf (stderr, "Opened audio device \"%s\"\n", (char *) name);

	// need to check the docs to see what the heck this actually means
	// I'm just glad it works... ;)
	tmp = 0x7fff0008;
	ioctl (fd, SNDCTL_DSP_SETFRAGMENT, &tmp);

	ioctl (fd, SNDCTL_DSP_GETOSPACE, &foo);

	LOG (LOG_INFO,
	     "fragments = %d\n fragstotal = %d\n fragsize = %d\n bytes = %d",
	     foo.fragments, foo.fragstotal, foo.fragsize, foo.bytes);

#endif
	return 0;
}

static int _audio_sys_close (void *plugin)
{
#if defined (__irix__)
	alClosePort (alport);
	alFreeConfig (alconfig);
	alport = 0;
	alconfig = 0;
	init = 0;

	return 0;
#else
	return close (fd);
#endif
}

static int _audio_sys_setup (plugin_output_audio_attr_t * attr)
{
	int ret = 0;

#if defined (__sun__)
/* Setup our parameters */
	audio_info_t info;

	AUDIO_INITINFO (&info);

	info.play.precision = attr->format;
	info.play.sample_rate = attr->speed;
	info.play.channels = attr->channels;
	//info.play.buffer_size = 1024;
	info.play.encoding = AUDIO_ENCODING_LINEAR;
	//info.play.port = AUDIO_SPEAKER;
	//info.play.gain = 110;

/* Write our configuration */
	if (ioctl (fd, AUDIO_SETINFO, &info) < 0) {
		fprintf (stderr, "%s: Writing audio config block\n",
			 strerror (errno));
		return -1;
	}
#else
	ret |=
	    ioctl (fd, SNDCTL_DSP_CHANNELS,
		   &attr->channels) ? AE_CHANNELS : 0;
	ret |=
	    ioctl (fd, SNDCTL_DSP_SETFMT, &attr->format) ? AE_FORMAT : 0;
	ret |= ioctl (fd, SNDCTL_DSP_SPEED, &attr->speed) ? AE_SPEED : 0;
#endif
	return -ret;
}

static int _audio_sys_write (const void *buf, size_t num_bytes)
{
	bufFill += num_bytes << 1;
#if defined (__irix__)
	alWriteFrames (alport, buf, num_bytes);
	return num_bytes;
#else
	return write (fd, buf, num_bytes);
#endif
}

static int _audio_sys_check (void)
{
#if defined (__sun__)
#else
	audio_buf_info buf_info;

	ioctl (fd, SNDCTL_DSP_GETOSPACE, &buf_info);

	if ((buf_info.fragstotal - buf_info.fragments) < 12)
		return 2;

	if ((buf_info.fragstotal - buf_info.fragments) < 36)
		return 1;
#endif

	return 0;
}


int PLUGIN_INIT (audio_sys) (char *whoami)
{
        pluginRegister (whoami,
                        PLUGIN_ID_OUTPUT_AUDIO,
                        NULL,
                        "sys",
                        NULL,
                        &audio_sys);

// TODO: check on open for AFMT_MPEG and AFMT_AC3
//      if not supported, card returns -EINVAL
//      if supported, use AFMT_QUERY

	return 0;
}

void PLUGIN_EXIT (audio_sys) (void)
{
}
