/*****
*
* This file is part of the OMS program.
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by 
* the Free Software Foundation; either version 2, or (at your option)
* any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; see the file COPYING.  If not, write to
* the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
*
*****/

/*
 *  video_out_sdl.c
 *
 *  Copyright (C) Dominik Schnitzer <aeneas@linuxvideo.org> - Janurary 13, 2001.
 *  Copyright (C) Ryan C. Gordon <icculus@lokigames.com> - April 22, 2000.
 *
 *  A mpeg2dec display driver that does output through the
 *  Simple DirectMedia Layer (SDL) library. This effectively gives us all
 *  sorts of output options: X11, SVGAlib, fbcon, AAlib, GGI. Win32, MacOS
 *  and BeOS support, too. Yay. SDL info, source, and binaries can be found
 *  at http://www.libsdl.org/
 *
 *  This file is part of oms, free DVD and Video player.
 *  It is derived from the mpeg2dec SDL video output plugin.
 *	
 *  oms is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  this oms output plugin is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 
 *
 *  Changes:
 *    Dominik Schnitzer <dominik@schnitzer.at> - November 08, 2000.
 *    - Added resizing support, fullscreen: chnaged the sdlmodes selection
 *       routine.
 *    - SDL bugfixes: removed the atexit(SLD_Quit), SDL_Quit now resides in
 *       the plugin_exit routine.
 *    - Commented the source :)
 *    - Shortcuts: for switching between Fullscreen/Windowed mode and for
 *       cycling between the different Fullscreen modes.
 *    - Small bugfixes: proper width/height of movie
 *    Dominik Schnitzer <dominik@schnitzer.at> - November 11, 2000.
 *    - Cleanup code, more comments
 *    - Better error handling
 *    Dominik Schnitzer <dominik@schnitzer.at> - December 17, 2000.
 *    - sdl_close() cleans up everything properly.
 *    - integrated Bruno Barreyra <barreyra@ufl.edu>'s fix which eliminates
 *       those memcpy()s for whole frames (sdl_draw_frame()) HACKMODE 2 patch
 *    - support for YV12 and IYUV format :>
 *    - minor fixups for future enhancements
 *    Dominik Schnitzer <dominik@schnitzer.at> - Janurary 13, 2001.
 *    - bugfix: Double buffered screenmodes made sdl output display nothing but
 *       a black screen.
 *    Dominik Schnitzer <dominik@schnitzer.at> - Janurary 20, 2001.
 *    - subpictures work! Properly! We even time the subpictures displayed they're
 *       still just blue, but it'll 100% work soon :)
 *    - switch between fullscreen and windowed mode now possible everytime, even
 *       when playback stopped
 *    - we keep the last frame displayed, so window content doesn't wash away
 *       when you i.e. resize the window when stopped. looks pretty cool.
 *    Dominik Schnitzer <dominik@schnitzer.at> - Janurary 21, 2001.
 *    - corrected some threading issues (now using mutexes)
 *    - chapter switching now works flawless on SDL side (!), _setup is checking
 *       if it was already called. no need to run _setup() twice.
 *
 *
 * Disable Xv SDL output
 *   export SDL_VIDEO_YUV_HWACCEL=0
 *
 * 
 */

/* define for debugging output */
/* #define DEBUG */

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <inttypes.h>

#include "oms/oms.h"
#include "oms/plugin/output_video.h"
#include "oms/log.h"
#include "alphablend.h"

#include <SDL/SDL.h>
#include <SDL/SDL_thread.h>

/* SDL Plugin status constants */
#define SDLSTATUS_CLOSED 0x0
#define SDLSTATUS_OPEN 0x1
#define SDLSTATUS_READY 0x2

/* Userdefined SDL Events */
#define SDLEVENT_VOQUIT SDL_USEREVENT


/** OMS Plugin functions **/

static int _sdl_open (void *plugin, void *name);
static int _sdl_close (void *plugin);
static int _sdl_setup (plugin_output_video_attr_t *attr);
static int _sdl_draw_frame (frame_t *src);
static int _sdl_draw_slice (uint8_t *src[], int slice_num);
static void _sdl_flip_page (void);
static int _sdl_overlay (overlay_buf_t *overlay_buf, int id);
static frame_t *_sdl_allocate_image_buffer (int width, int height, uint32_t format);
static void _sdl_free_image_buffer (frame_t *image);


/** Private SDL Data structure **/

static struct sdl_priv_s {
	/* current subtitles hack */
	overlay_buf_t overlay_buf;
	SDL_TimerID overlay_timerid;
	int overlay_visible;

	/* SDL YUV surface & overlay */
	SDL_Surface *surface;
	SDL_Overlay *overlay, *current_frame;
	
	/* available fullscreen modes */
	SDL_Rect **fullmodes;
	
	/* event checking tread */
	SDL_Thread *eventcheck;
	SDL_mutex *eventcheck_mutex;
	
	/* surface attributes for fullscreen and windowed mode */
	Uint32 sdlflags, sdlfullflags;

	/* save the windowed output extents */
	SDL_Rect windowsize, imagesize;
	
	/* Bits per Pixel */
	Uint8 bpp;

	/* current fullscreen mode, 0 = highest available fullscreen mode */
	int fullmode;

	/* SDL vo status */
	int status;

	/* YUV ints */
	int framePlaneY, framePlaneUV;
	int slicePlaneY, slicePlaneUV;
} _sdl_priv;


LIBVIDEO_EXTERN(_sdl,"sdl");


/** libvo Plugin functions **/

/**
 * Take a null-terminated array of pointers, and find the last element.
 *
 *    params : array == array of which we want to find the last element.
 *   returns : index of last NON-NULL element.
 **/

static inline int findArrayEnd (SDL_Rect **array)
{
	int i = 0;
	while ( array[i++] );	/* keep loopin' ... */
	
	/* return the index of the last array element */
	return i - 1;
}


/**
 * Open and prepare SDL output.
 *
 *    params : *plugin ==
 *             *name == 
 *   returns : 0 on success, -1 on failure
 **/

static int _sdl_open (void *plugin, void *name)
{
	struct sdl_priv_s *priv = &_sdl_priv;
	const SDL_VideoInfo *vidInfo = NULL;
	
	LOG (LOG_DEBUG, "SDL video out: Opening Plugin");
	
	/* Check if the sdl plugin was initialized */
	if (priv->status & SDLSTATUS_OPEN) {
		LOG (LOG_INFO, "SDL video out: SDL was already initialized. (sdl_open() already called). I'll continue anyway.");
		return 0;
	}

	/* Cleanup YUV Overlay structures */
	priv->overlay = NULL;
	priv->current_frame = NULL;
	priv->surface = NULL;
	priv->fullmodes = NULL;
	priv->overlay_buf.data = NULL;
	priv->overlay_visible = 0;
	
	/* Reset the configuration vars to its default values */
	priv->bpp = 0;
	priv->fullmode = -2;
	priv->sdlflags = SDL_HWSURFACE|SDL_RESIZABLE;
	priv->sdlfullflags = SDL_HWSURFACE|SDL_FULLSCREEN;

	if (priv->status & SDLSTATUS_OPEN) {
		LOG (LOG_DEBUG, "SDL video out: Plugin already open!");
		return 0;
	}

	/* initialize the SDL Video system */
	if (SDL_Init (SDL_INIT_VIDEO)) {
		LOG (LOG_ERROR, "SDL video out: Initializing of SDL failed (SDL_Init). Please use the latest version of SDL.");
		return -1;
	}
	/* Timer initialization */
	if (SDL_InitSubSystem (SDL_INIT_TIMER)) {
		LOG (LOG_WARNING, "SDL video out: Initializing the timer failed. Overlays (i.e. subtitles) wont be displayed right.");
		return -1;
	}
		
	/* No Keyrepeats! */
	SDL_EnableKeyRepeat(0,0);

	/* get information about the graphics adapter */
	vidInfo = SDL_GetVideoInfo ();
	
	/* collect all fullscreen & hardware modes available */
	if (!(priv->fullmodes = SDL_ListModes (vidInfo->vfmt, priv->sdlfullflags))) {

		/* non hardware accelerated fullscreen modes */
		priv->sdlfullflags &= ~SDL_HWSURFACE;
 		priv->fullmodes = SDL_ListModes (vidInfo->vfmt, priv->sdlfullflags);
	}
	
	/* test for normal resizeable & windowed hardware accellerated surfaces */
	if (!SDL_ListModes (vidInfo->vfmt, priv->sdlflags)) {
		
		/* test for NON hardware accelerated resizeable surfaces - poor you. 
		 * That's all we have. If this fails there's nothing left.
		 * Theoretically there could be Fullscreenmodes left - we ignore this for now.
		 */
		priv->sdlflags &= ~SDL_HWSURFACE;
		if ((!SDL_ListModes (vidInfo->vfmt, priv->sdlflags)) && (!priv->fullmodes)) {
			LOG (LOG_ERROR, "SDL video out: Couldn't get any acceptable SDL Mode for output. (SDL_ListModes failed)");
			return -1;
		}
	}
	
		
   /* YUV overlays need at least 16-bit color depth, but the
    * display might less. The SDL AAlib target says it can only do
    * 8-bits, for example. So, if the display is less than 16-bits,
    * we'll force the BPP to 16, and pray that SDL can emulate for us.
	 */
	priv->bpp = vidInfo->vfmt->BitsPerPixel;
	if (priv->bpp < 16) {
		LOG (LOG_WARNING, "SDL video out: Your SDL display target wants to be at a color depth of (%d), but we need it to be at\
least 16 bits, so we need to emulate 16-bit color. This is going to slow things down; you might want to\
increase your display's color depth, if possible", priv->bpp);
		priv->bpp = 16;  
	}
	
	/* We dont want those in our event queue */
	SDL_EventState(SDL_ACTIVEEVENT, SDL_IGNORE);
	SDL_EventState(SDL_KEYUP, SDL_IGNORE);
	SDL_EventState(SDL_MOUSEBUTTONDOWN, SDL_IGNORE);
	SDL_EventState(SDL_MOUSEBUTTONUP, SDL_IGNORE); 
	SDL_EventState(SDL_QUIT, SDL_IGNORE);
	SDL_EventState(SDL_SYSWMEVENT, SDL_IGNORE);
	
	/* Success! */
	priv->status = priv->status | SDLSTATUS_OPEN;
	LOG (LOG_DEBUG, "SDL video out: Opened plugin.");
	return 0;
}


/**
 * Close SDL, Cleanups, Free Memory
 *
 *    params : *plugin
 *   returns : non-zero on success, zero on error.
 **/

static int _sdl_close (void *plugin)
{
	struct sdl_priv_s *priv = &_sdl_priv;
	SDL_Event event_voquit;

	LOG (LOG_DEBUG, "SDL video out: Closing Plugin");
	
	/* Stop the event loop */
	event_voquit.type = SDLEVENT_VOQUIT;
	SDL_PushEvent(&event_voquit);

	/* Check if the sdl plugin was initialized */
	if (priv->status == SDLSTATUS_CLOSED) {
		LOG (LOG_INFO, "SDL video out: SDL Is not opened. No need to continue sdl_close().");
		return 0;
	}
	
	/* free data if it's still valid and in overlay_buf.data */
	if (priv->overlay_buf.data)
		free (priv->overlay_buf.data);
	
	/* Cleanup YUV Overlay structure */
	if (priv->overlay) 
		SDL_FreeYUVOverlay(priv->overlay);

	/* Free our blitting surface */
	if (priv->surface)
		SDL_FreeSurface(priv->surface);
	
	/* DONT attempt to free the fullscreen modes array. SDL_Quit* does this for us */
	/* Cleanup SDL */
	SDL_QuitSubSystem(SDL_INIT_VIDEO);
	
	/* Dealloc the Timer */
	SDL_QuitSubSystem(SDL_INIT_TIMER);

	/* We've closed SDL vo */
	priv->status = SDLSTATUS_CLOSED;

	LOG (LOG_DEBUG, "SDL video out: Closed Plugin");

	return 0;
}


/**
 * Sets the specified fullscreen mode.
 *
 *   params : mode == index of the desired fullscreen mode
 *  returns : doesn't return
 **/
 
static void set_fullmode (int mode)
{
	struct sdl_priv_s *priv = &_sdl_priv;
	SDL_Surface *newsurface = NULL;
	
	
	/* if we haven't set a fullmode yet, default to the lowest res fullmode first */
	if (mode < 0) 
		mode = priv->fullmode = findArrayEnd(priv->fullmodes) - 1;

	/* change to given fullscreen mode and hide the mouse cursor*/
	newsurface = SDL_SetVideoMode(priv->fullmodes[mode]->w, priv->fullmodes[mode]->h, priv->bpp, priv->sdlfullflags);
	
	/* if we were successfull hide the mouse cursor and save the mode */
	if (newsurface)
		priv->surface = newsurface;
}


/**
 * Checks for SDL keypress and window resize events, this function
 * runs in a seperate event-checking thread, "eventcheck"
 *
 *   params : none
 *  returns : doesn't return
 **/

static void check_events (void)
{
	struct sdl_priv_s *priv = &_sdl_priv;
	int leave = 0;
	Uint32 mouse_visibletime = 0;
	int mouse_visible = 1;
	SDL_Event event, event_voquit;
	SDLKey keypressed; 
	
	
	LOG (LOG_DEBUG, "SDL video out: Eventcheck Thread started");

	event_voquit.type = SDLEVENT_VOQUIT;	
	
	/* Poll the waiting SDL Events */
	while (!leave) {

	/* dont let the thread block all CPU time */	
	SDL_Delay(150);
	
	/* check if we can fade out the mouse */
	if ((mouse_visible) && (SDL_GetTicks() - mouse_visibletime > 4000)) {
		LOG (LOG_INFO, "SDL video out: Hide Cursor");
		SDL_ShowCursor(0);
		mouse_visible = 0;
		LOG (LOG_INFO, "SDL video out: Hide Cursor");
	}
	
	/* halt output, till we're finished here */		
	if (SDL_LockMutex(priv->eventcheck_mutex) == -1)
		LOG (LOG_WARNING, "SDL video out: Could not Lock mutex. There is a threading problem. Strange things will happen now");
	
	while (SDL_PollEvent(&event)) {

		switch (event.type) {

			/* capture window resize events */
			case SDL_VIDEORESIZE:
				priv->surface = SDL_SetVideoMode(event.resize.w, event.resize.h, priv->bpp, priv->sdlflags);

				/* save video extents, to restore them after going fullscreen */
				priv->windowsize.w = priv->surface->w;
				priv->windowsize.h = priv->surface->h;
				
				LOG (LOG_DEBUG, "SDL video out: Window resize (%dx%d)", priv->windowsize.w, priv->windowsize.h);
			break;
			
			
			/* graphics more selection shortcuts */
			case SDL_KEYDOWN:
				keypressed = event.key.keysym.sym;
				
				/* plus key pressed. plus cycles through available fullscreenmodes, if we have some */
				if ( ((keypressed == SDLK_PLUS) || (keypressed == SDLK_KP_PLUS)) && (priv->fullmodes) ) {
					/* select next fullscreen mode */
					priv->fullmode++;
					if (priv->fullmode > (findArrayEnd(priv->fullmodes) - 1)) priv->fullmode = 0;
					set_fullmode(priv->fullmode);
	
					LOG (LOG_DEBUG, "SDL video out: Set next available fullscreen mode.");
				}
				
				if (keypressed == SDLK_0) {
					SDL_PushEvent(&event_voquit);
				}

				/* return or escape key pressed toggles/exits fullscreenmode */
				else if ( (keypressed == SDLK_RETURN) || (keypressed == SDLK_ESCAPE) ) {
				 	if (priv->surface->flags & SDL_FULLSCREEN) {
						priv->surface = SDL_SetVideoMode(priv->windowsize.w, priv->windowsize.h, priv->bpp, priv->sdlflags);
						SDL_ShowCursor(1);
						
						LOG (LOG_DEBUG, "SDL video out: Windowed mode");
					} 
					else if (priv->fullmodes){
						set_fullmode(priv->fullmode);

						LOG (LOG_DEBUG, "SDL video out: Set fullscreen mode.");
					}
				}
				break;

			/* graphics more selection shortcuts */
			case SDLEVENT_VOQUIT:
				leave = 1;
			break;
			
			/* Make the mouse visible on mousemove */
			case SDL_MOUSEMOTION:
				mouse_visibletime = SDL_GetTicks();
				if (!mouse_visible) {
					SDL_ShowCursor(1);
					mouse_visible = 1;
				}
			break;
				
		}
		
	}

	if ((oms_get_status() == STATUS_STOP) || (oms_get_status() == STATUS_PAUSE))
		if (priv->overlay)
			SDL_DisplayYUVOverlay (priv->overlay, &priv->surface->clip_rect);
			
	/* output can continue */
	if (SDL_UnlockMutex(priv->eventcheck_mutex) == -1)
		LOG (LOG_WARNING, "SDL video out: Could not unlock mutex, looks like a threading problem");
		

	}

			
	LOG (LOG_DEBUG, "SDL video out: Eventcheck Thread finished");
	
}


/**
 * Initialize an SDL surface and an SDL YUV overlay.
 *
 *    params : width  == width of video we'll be displaying.
 *             height == height of video we'll be displaying.
 *             fullscreen == want to be fullscreen?
 *             title == Title for window titlebar.
 *   returns : non-zero on success, zero on error.
 **/

static int _sdl_setup (vo_output_video_attr_t *attr)
{
	struct sdl_priv_s *priv = &_sdl_priv;
	
	LOG (LOG_DEBUG, "SDL video out: Setup started");

	/* Check if the sdl plugin was initialized */
	if (!(priv->status & SDLSTATUS_OPEN)) {
		LOG (LOG_ERROR, "SDL video out: SDL was not initalized. (I'm missing the sdl_open() call)");
		return -1;
	}
	
	/* if some crazy guy calls setup 2 times... we'll check here*/
	if  (priv->status & SDLSTATUS_READY) {
		LOG (LOG_INFO, "SDL video out: Already ready for video output. (Skipping _setup() call this time) Everything is O.K.");
		return 0;
	}

	/* Set window title of our output window */
	if (attr->title)
		SDL_WM_SetCaption (attr->title, "SDL Video Out");
	else
		SDL_WM_SetCaption ("SDL: RETURN/ESCAPE = Toggle Fullscreen/Window; PLUS = Cycle Fullscreen Resolutions", "SDL Video Out");

	/* Set the video mode via SDL & check for Fullscreen first */
	if ((attr->fullscreen) && priv->fullmodes) {
		if (!(priv->surface = SDL_SetVideoMode (priv->fullmodes[0]->w, priv->fullmodes[0]->h, priv->bpp, priv->sdlfullflags))) {
			LOG (LOG_ERROR, "SDL video out: Could not set the desired fullscreen mode (SDL_SetVideoMode). Will try windowed mode next.");

			/* disable fullscreen mode and free memory */
			attr->fullscreen = 0;
		}
	}

	/* Initialize the windowed SDL Window per default */
	if (!attr->fullscreen)
		if (!(priv->surface = SDL_SetVideoMode (attr->width, attr->height, priv->bpp, priv->sdlflags))) {
			LOG (LOG_ERROR, "SDLvideo out: Could not set the desired video mode (SDL_SetVideoMode)");
			return -1;
		}
	
	/* Initialize and create the YUV Overlay used for video out */
	if (!(priv->overlay = SDL_CreateYUVOverlay (attr->width, attr->height, SDL_YV12_OVERLAY, priv->surface))) {
		LOG (LOG_ERROR, "SDL video out: Couldn't create an SDL-based YUV overlay");
		return -1;
	}
	priv->framePlaneY = (attr->width * attr->height);
	priv->framePlaneUV = ((attr->width / 2) * (attr->height / 2));
	priv->slicePlaneY = ((attr->width) * 16);
	priv->slicePlaneUV = ((attr->width / 2) * (8));
	
	/* Save the original Image size */
	priv->windowsize.w = attr->width;
	priv->windowsize.h = attr->height;
	priv->imagesize.w = attr->width;
	priv->imagesize.h = attr->height;
	
	priv->status = priv->status | SDLSTATUS_READY;
	
	priv->eventcheck_mutex = SDL_CreateMutex();
	priv->eventcheck = SDL_CreateThread( (void *) check_events, NULL);
	
	LOG (LOG_DEBUG, "SDL video out: Setup ended. bpp: %d", priv->bpp);

	return 0;
}


/**
 * Draw a frame to the SDL YUV overlay.
 *
 *   params : *src[] == the Y, U, and V planes that make up the frame.
 *  returns : non-zero on success, zero on error.
 **/

static int _sdl_draw_frame (frame_t *frame)
{
	struct sdl_priv_s *priv = &_sdl_priv;
	
	priv->current_frame = (SDL_Overlay*) frame->private;
	
	/* Unlock the frame - the frame is 100% filled with data to display 
	 * We Lock it again when the frame was displayed. */
	SDL_UnlockYUVOverlay (priv->current_frame);

	return 0;
}


/**
 * Draw a slice (16 rows of image) to the SDL YUV overlay.
 *
 *   params : *src[] == the Y, U, and V planes that make up the slice.
 *  returns : non-zero on error, zero on success.
 *
 *  FIXME: priv->overlay when we check for events needs to be locked
 **/

static int _sdl_draw_slice (uint8_t *src[], int slice_num)
{
	struct sdl_priv_s *priv = &_sdl_priv;
	uint8_t *dst;

	/* since we haven't allocated memory for  */
	priv->current_frame = priv->overlay;

	if (SDL_LockYUVOverlay (priv->overlay)) {
		LOG (LOG_ERROR, "SDL video out: Couldn't lock SDL-based YUV overlay");
		return -1;
	}

	dst = (uint8_t *) *(priv->overlay->pixels) + (priv->slicePlaneY * slice_num);
	memcpy (dst, src[0], priv->slicePlaneY);
	dst = (uint8_t *) *(priv->overlay->pixels) + priv->framePlaneY + (priv->slicePlaneUV * slice_num);
	memcpy (dst, src[1], priv->slicePlaneUV);
	dst += priv->framePlaneUV;
	memcpy (dst, src[2], priv->slicePlaneUV);
	
	SDL_UnlockYUVOverlay (priv->overlay);

	return 0;
}


/**
 * Display the surface we have written our data to and check for events.
 *
 *   params : mode == index of the desired fullscreen mode
 *  returns : doesn't return
 **/

static void _sdl_flip_page (void)
{
	struct sdl_priv_s *priv = &_sdl_priv;
	
	/* eventchecking should stop */
	if (SDL_LockMutex(priv->eventcheck_mutex) == -1)
		LOG (LOG_WARNING, "SDL video out: Could lock mutex for threadsafe blitting.");
	
	if (priv->current_frame == NULL) {
		LOG(LOG_ERROR, "SDL video out: This shouldn't happen! current_frame is NULL!");
		return;
	}
	
	SDL_LockYUVOverlay (priv->overlay);
	memcpy(priv->overlay->pixels[0], priv->current_frame->pixels[0], priv->imagesize.w * priv->imagesize.h * 1.5);
	/* spu hack - overlay out spu things */
	if ((priv->overlay_buf.data) && (priv->overlay_visible)){
		blend_yuv (priv->overlay->pixels[0], &priv->overlay_buf, priv->imagesize.w, priv->imagesize.h);
	}
	SDL_UnlockYUVOverlay (priv->overlay);
	
	/* blit to the YUV overlay */
	SDL_DisplayYUVOverlay (priv->overlay, &priv->surface->clip_rect);

	/* eventchecking can continue */
	if (SDL_UnlockMutex(priv->eventcheck_mutex) == -1)
		LOG (LOG_WARNING, "SDL video out: Could not unlock blitting mutex.");

	/* lock the frame again, since we've just displayed the frame. we're
	* ready for new data */
	SDL_LockYUVOverlay (priv->current_frame);
}


/**
 * grab overlay data, used for subtitles, we display them in filp()
 *
 *   params : overlay_buf == buffer containing overlay data and information
 *            id == unique id of the overlay, currently there's just one overlay used
 *             and this is for the subtitles.
 *  returns : 0 on success
 **/

static void overlay_remove ()
{
	struct sdl_priv_s *priv = &_sdl_priv;
	
	/* free data if it's still valid and in overlay_buf.data */
	if (priv->overlay_buf.data)
		priv->overlay_visible = 0;
}


/**
 * grab overlay data, used for subtitles, we display them in filp()
 *
 *   params : overlay_buf == buffer containing overlay data and information
 *            id == unique id of the overlay, currently there's just one overlay used
 *             and this is for the subtitles.
 *  returns : 0 on success
 **/

static int _sdl_overlay (overlay_buf_t *overlay_buf, int id)
{
	struct sdl_priv_s *priv = &_sdl_priv;

	/* free data if it's still valid and in overlay_buf.data */
	if (priv->overlay_buf.data) {
		free (priv->overlay_buf.data);
		priv->overlay_buf.data = NULL;
		priv->overlay_visible = 0;
	}
	
	/* grab the overlay data disabled subpictures, since blend_yuv() doesn't work yet */
	memcpy (&priv->overlay_buf, overlay_buf, sizeof(overlay_buf_t));
	
	/* Set Timers */
	SDL_RemoveTimer(priv->overlay_timerid);
	priv->overlay_timerid = SDL_AddTimer(overlay_buf->time_execute, (void *) overlay_remove, NULL);

	/* Make subpictures visible again */
	priv->overlay_visible = 1;

	return 0;
}


/**
 * Allocate an SDL frame buffer. Called once to allocate memory for a frame.
 *
 *   params : width == needed surface width
 *            height == needed surface height
 *            format == image format
 *  returns : the frame ready to fill with image data
 **/

static frame_t *_sdl_allocate_image_buffer (int width, int height, uint32_t format)
{
	struct sdl_priv_s *priv = &_sdl_priv;
	frame_t	*frame;

	LOG (LOG_DEBUG, "SDL video out: allocate image buffer");

	if (!(frame = malloc (sizeof (frame_t))))
		return NULL;

	/* This is a bug somewhere else height == width, and width == height */
	if (!(frame->private = (void*) SDL_CreateYUVOverlay (height, width,
			format, priv->surface)))
	{
		LOG (LOG_ERROR, "SDL video out: Couldn't create an SDL-based YUV overlay");
		return NULL;
	}

	/* do we have a YV12 (Y + V + U) format 
	 * base[0] = Y
	 * base[1] = U
	 * base[2] = V */
	if (format == SDL_YV12_OVERLAY) {
		frame->base[0] = (uint8_t*) ((SDL_Overlay*) (frame->private))->pixels[0];
		frame->base[2] = (uint8_t*) ((SDL_Overlay*) (frame->private))->pixels[1];
		frame->base[1] = (uint8_t*) ((SDL_Overlay*) (frame->private))->pixels[2];
		LOG (LOG_DEBUG, "SDL video out: Allocated a YV12 Overlay (YVU: %x)", format);
	} else

	/* check if we have a classic IYUV format (Y + U + V) */
	if (format == SDL_IYUV_OVERLAY) {
		frame->base[0] = (uint8_t*) ((SDL_Overlay*) (frame->private))->pixels[0];
		frame->base[1] = (uint8_t*) ((SDL_Overlay*) (frame->private))->pixels[1];
		frame->base[2] = (uint8_t*) ((SDL_Overlay*) (frame->private))->pixels[2];
		LOG (LOG_DEBUG, "SDL video out: Allocated a YUV Overlay (YUV: %x)", format);
	} else {

		LOG (LOG_ERROR, "SDL video out: Unknown YUV Image Format: %x", format);
		return NULL;
	}

	/* Locks the allocated frame, to allow writing to it.
	 * sdl_flip Unlocks it. sdl_draw_frame Locks it again.*/
	
	SDL_LockYUVOverlay ((SDL_Overlay*) frame->private);
	
	return frame;
}


/**
 * Free an SDL image frame from memory
 *
 *   params : frame == frame to 
 *  returns : doesn't return
 **/

static void _sdl_free_image_buffer(frame_t* frame)
{
	LOG (LOG_DEBUG, "SDL video out: free image buffer");

	SDL_FreeYUVOverlay((SDL_Overlay*) frame->private);
	free(frame);
	frame = NULL;
}


int PLUGIN_INIT(vo_sdl) (char *whoami)
{
	pluginRegister (whoami,
		PLUGIN_ID_OUTPUT_VIDEO,
		NULL,
		"sdl",
		NULL,
		&video_out_sdl);

	return 0;
}


void PLUGIN_EXIT(vo_sdl) (void)
{
	_sdl_close(NULL);
}
